"""
Class to read the Fermi catalogs
 author David Sanchez  david.sanchez@lapp.in2p3.fr
"""

import string,numpy,os
import astropy.io.fits as fits
import PlotLibrary
# from environ import VERSION_4FGL,VERSION_3FGL,VERSION_2FGL,VERSION_1FHL,VERSION_2FHL,VERSION_3FHL,FERMI_CATALOG_DIR
from math import *
import pandas as pd
# import ReadMWLCatalog


from astropy.coordinates import ICRS, Galactic, FK4, FK5, SkyCoord

import sys

class base(object):
    def __init__(self):
        self.classname = self.__class__.__name__
        self.errorcolor = "\033[31m"#red
        self.infocolor = "\033[34m"#blue
        self.warningcolor = "\033[33m"#yellow
        self.successcolor = "\033[32m"#green
        self.endcolor = "\033[0m"#reset
        self.prependfunction = True
        self.basemembers = ["classname","errorcolor","infocolor","warningcolor","successcolor","endcolor","prependfunction"]
        
    def error(self,message, functionname = ""):
        printstring = ""
        if functionname == "":
            printstring = "\n"+self.errorcolor+"*** Error ["+self.classname+"]: "+message+" ***\n"+self.endcolor
        else:
            printstring = "\n"+self.errorcolor+"*** Error ["+self.classname+"::"+functionname+"]: "+message+" ***\n"+self.endcolor
        sys.exit(printstring)
    
    def info(self,message,newline=True):
        printstring = self.infocolor+"["+self.classname+"]: "+message+self.endcolor     
        if newline:
            print (printstring)
        else:
            print (self.infocolor+message+self.endcolor)
            sys.stdout.flush()  
        
    def warning(self,message,functionname = ""):
        printstring = ""
        if functionname == "":
            printstring = self.warningcolor+"["+self.classname+"] Warning: "+message+self.endcolor
        else:
            printstring = self.warningcolor+"["+self.classname+"::"+functionname+"] Warning: "+message+self.endcolor
        print (printstring)
        
    def success(self,message):
        printstring = self.successcolor+"["+self.classname+"]: "+message+self.endcolor
        print (printstring)
        
    def progress(self,message="."):
        string = self.infocolor+message+self.endcolor
        print (string)
        

class FermiCatalogReader(base):
  ''' Class to read the Fermi Catalogs '''
  def __init__(self,name=None,Representation = "e2dnde",escale = "TeV"):
    ''' init function
    Parameters
    ---------
    name    : catalog name of the source
    folder  : where the Fermi catalog are. If None, the FERMI_CATALOG_DIR environnement variable is used
    Representation : for the plot (dnde, ednde, e2dnde)
    escale  : energy scale in MeV, GeV or TeV
    '''
    super(FermiCatalogReader,self).__init__()
    base.__init__(self)


    self.Representation = Representation
    self.escale = escale

    #Check the input
    supportedRep = ["dnde","ednde","e2dnde"]
    if not(Representation in supportedRep):
        raise RuntimeError("Representation not supported")
    supportedEscale = ["MeV","GeV","TeV"]
    if not(escale in supportedEscale):
        raise RuntimeError("Escale not supported")


    # list of supported catalog
    self.list_of_catalog = ['4FGL','3FGL','2FGL']
    #all the information are in a python dictionnary
    self.CatalogData = {}
    for cat in self.list_of_catalog:
      self.CatalogData[cat] = {}
    self.CatalogData['4FGL']['fits'] = "gll_psc_v31.fit"
    self.CatalogData['3FGL']['fits'] = "gll_psc_v16.fit"
    self.CatalogData['2FGL']['fits'] = "gll_psc_v09.fit"

    self.CatalogData['4FGL']['data'] = fits.open(self.CatalogData['4FGL']['fits'])[1].data
    self.CatalogData['3FGL']['data'] = fits.open(self.CatalogData['3FGL']['fits'])[1].data
    self.CatalogData['2FGL']['data'] = fits.open(self.CatalogData['2FGL']['fits'])[1].data

    #read data points
    self.CatalogData['4FGL']['Band'] = ['Flux_Band','Flux_Band','Flux_Band','Flux_Band','Flux_Band','Flux_Band','Flux_Band']
    self.CatalogData['3FGL']['Band'] = ['Flux100_300','Flux300_1000','Flux1000_3000','Flux3000_10000','Flux10000_100000']
    self.CatalogData['2FGL']['Band'] = ['Flux100_300','Flux300_1000','Flux1000_3000','Flux3000_10000','Flux10000_100000']

    #read data points Significance
    self.CatalogData['4FGL']['BandTS'] = ['Sqrt_TS_Band','Sqrt_TS_Band','Sqrt_TS_Band','Sqrt_TS_Band','Sqrt_TS_Band','Sqrt_TS_Band','Sqrt_TS_Band']
    self.CatalogData['3FGL']['BandTS'] = ['Sqrt_TS100_300','Sqrt_TS300_1000','Sqrt_TS1000_3000','Sqrt_TS3000_10000','Sqrt_TS10000_100000']
    self.CatalogData['2FGL']['BandTS'] = ['Sqrt_TS100_300','Sqrt_TS300_1000','Sqrt_TS1000_3000','Sqrt_TS3000_10000','Sqrt_TS10000_100000']

    #upper bound of the energy bins of Fermi catalogs
    self.CatalogData['4FGL']['eMax'] = self._HandleEnergyUnit(numpy.array([100,300,1000,3000,10000,30000,300000]))
    self.CatalogData['3FGL']['eMax'] = self._HandleEnergyUnit(numpy.array([300,1000,3000,10000,100000]))
    self.CatalogData['2FGL']['eMax'] = self._HandleEnergyUnit(numpy.array([300,1000,3000,10000,100000]))

    #lower bound of the energy bins of Fermi catalogs
    self.CatalogData['4FGL']['eMin'] = self._HandleEnergyUnit(numpy.array([50,100,300,1000,3000,10000,30000]))
    self.CatalogData['3FGL']['eMin'] = self._HandleEnergyUnit(numpy.array([100,300,1000,3000,10000]))
    self.CatalogData['2FGL']['eMin'] = self._HandleEnergyUnit(numpy.array([100,300,1000,3000,10000]))

    self.name = name
    self.Spec = None

    self.info("creating catalogues Reader with\n\t ")
    for cat in self.list_of_catalog:
      print( self.CatalogData[cat]['fits']+"\n\t ")

    # get table indices corresponding to the source entry
    if name != None:
        self.GetIndices()
    
    
  @classmethod
  def fromName(self,name, frame = FK5,Representation = "e2dnde",escale = "TeV"):
    ''' return a FermiCatalogReader object based on a name of a source
    Parameters
    ----------
    name    : catalog name (see astropy manual for the valid names)
    frame   : Astropy coordinate frame ICRS, Galactic, FK4, FK5 , see astropy for more information
    folder  : where the Fermi catalog are
    Representation : for the plot (dnde, ednde, e2dnde)
    energy scale : MeV, GeV or TeV

    '''
    coord =  SkyCoord.from_name(name,frame)
    catalog = FermiCatalogReader(None)
    for k in catalog.list_of_catalog:
        catalog.findfromCoordinate(k,coord.ra.deg,coord.dec.deg)
        if catalog.CatalogData[k]['found']:
            break
        
    return FermiCatalogReader(catalog.CatalogData[k]['name'],Representation = Representation, escale = escale)


  def GetCatalogName(self,key):
    '''
      return the fermi name of catalo key
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    name=self.CatalogData[key]['data'].field('Source_Name')
    try :
      indice = self.CatalogData[key]['indice']
    except :
      self.error("cannot find source named in the "+key)
    return name[indice]


  def GetIndices(self):
    ''' look for the table indices where the source data are in the catalog'''
    for k in self.CatalogData.keys():
      name=self.CatalogData[k]['data'].field('Source_Name')
      print ('try ',k,' catalog')
      for indice in range(name.size):
        if name[indice] == self.name :
          self.CatalogData[k]['indice'] = indice
          self.CatalogData[k]['name']   = name[indice]
          self.success(k+" source found")
          self.CatalogData[k]['found'] = True

          ra,dec = self.GetPosition(k)
          self.info("Catalog Position "+str(ra)+","+str(dec))
          for j in self.CatalogData.keys():
            if j!=k:
              self.findfromCoordinate(j,ra,dec)

          self.GetModels()
          self.GetClass()
          return
    self.error("cannot find the index of the source named "+self.name+". Check the name")

  def GetPosition(self,k,frame = "radec"):
    '''retrive the position of the source in J2000 '''
    if frame == "radec":
      ra  = self.CatalogData[k]['data'].field('RAJ2000')
      dec = self.CatalogData[k]['data'].field('DEJ2000')
    elif frame == "galactic":
      ra  = self.CatalogData[k]['data'].field('GLON')
      dec = self.CatalogData[k]['data'].field('GLAT')
    return ra[self.CatalogData[k]['indice']],dec[self.CatalogData[k]['indice']]



  def findfromCoordinate(self, k,ra0,dec0):
    '''
    find a source based on coordinates
    Parameters
    ----------
    k   : name of the catalog 2FGL, 3FGL, etc...
    ra0 : coordinate of the source in J2000. The column read is RAJ2000
    dec0 : coordinate of the source in J2000. The column read is DEJ2000
    '''
    ra  = self.CatalogData[k]['data'].field('RAJ2000')
    dec = self.CatalogData[k]['data'].field('DEJ2000')

    r = calcAngSepDeg(ra,dec,ra0,dec0)
    res = r.argmin()
    if min(r) > 0.5:
      self.warning("No source closer than 0.5 degrees in catalog ",k)
      self.CatalogData[k]['found'] = False

      return
    self.success("found a close source in the catalog "+self.CatalogData[k]['fits']+" at r="+str(r[res])+" named "+self.CatalogData[k]['data'].field('Source_Name')[res])
    self.CatalogData[k]['indice'] = res
    self.CatalogData[k]['name'] = self.CatalogData[k]['data'].field('Source_Name')[res]
    self.CatalogData[k]['found'] = True


  def GetModels(self):
    '''retrive the spectral model in the catalog and fill the dictionnary'''
    for k in self.CatalogData.keys():
      if self.CatalogData[k]['found'] == False:
        continue
      try:
        Spec = self.CatalogData[k]['data'].field('SpectrumType')
        self.CatalogData[k]['model'] = Spec[self.CatalogData[k]['indice']]
      except:
        self.CatalogData[k]['model'] = "PowerLaw"

      self.info(k+" model type: "+self.CatalogData[k]['model'])

  def GetClass(self):
    '''
    retrive the object class from the CLASS1 field and fill the dictionnary
    '''
    for k in self.CatalogData.keys():
      if self.CatalogData[k]['found'] == False:
        continue
      try : 
         Class = self.CatalogData[k]['data'].field('CLASS1')
      except : 
         Class = self.CatalogData[k]['data'].field('CLASS') #for the 2FHL
      self.CatalogData[k]['class'] = Class[self.CatalogData[k]['indice']]
      self.info(k+" Object class: "+self.CatalogData[k]['class'])

  def ReturnClass(self,key):
    if self.CatalogData[key]['found'] == False:
      return "unknow"
    try : 
       return self.CatalogData[key]['data'].field('CLASS1')[self.CatalogData[key]['indice']]
    except : 
       return self.CatalogData[key]['data'].field('CLASS')[self.CatalogData[key]['indice']] #for the 2FHL

  def GetDataPoints(self,key):
    ''' read data points and fill the dictionnary
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''

    # if key == "4FGL":
      # self.warning("No data points yet  in catalog: "+key)
      # return
    flux  = []
    dflux = []
    try:
      if self.CatalogData[key]['found'] == False: 
        self.error("This source does not belong to "+key)
        return 0,0,0,0
      self.info("Read data points for "+key)
      for i in range(len(self.CatalogData[key]['Band'])):
          if key == '3FHL' or key == "4FGL":
            flux.append(self.CatalogData[key]['data'].field(self.CatalogData[key]['Band'][i])[self.CatalogData[key]['indice']][i])
          else:
            flux.append(self.CatalogData[key]['data'].field(self.CatalogData[key]['Band'][i])[self.CatalogData[key]['indice']])
          
          if key == '3FHL' or key == '4FGL' or key == '3FGL' or key == '1FHL' or key == '2FHL':
            if key == '3FHL' or key == '4FGL' :
              tmp = self.CatalogData[key]['data'].field("Unc_"+self.CatalogData[key]['Band'][i])[self.CatalogData[key]['indice']][i]
            else:
              tmp = self.CatalogData[key]['data'].field("Unc_"+self.CatalogData[key]['Band'][i])[self.CatalogData[key]['indice']]
            try :
              TS = self.CatalogData[key]['data'].field(self.CatalogData[key]['BandTS'][i])[self.CatalogData[key]['indice']]
              if key == '3FHL' or key == "4FGL":
                TS = TS[i]
            except:
              self.warning("No TS found in catalog: "+key)
            if TS <4:
              dflux.append(0)
            else:
              dflux.append(abs(tmp[0]))
            # dflux.append(max(tmp[0],tmp[1]))
          else:
            dflux.append(self.CatalogData[key]['data'].field("Unc_"+self.CatalogData[key]['Band'][i])[self.CatalogData[key]['indice']])

      return self.CatalogData[key]['eMin'],self.CatalogData[key]['eMax'],numpy.array(flux),numpy.array(dflux)
    except :
      self.error("problem in GetDataPoints function for catalog : "+key)


#########################
  def MakeSpectrum(self,key,Emin=100,Emax=3e5):
    '''
    Depending on the found spectral model, this function will call another function that
    read the catalog and return the data in a format readable by the plot library.
    A spectrum object  (from the plotlibrary) is then created and stored in the python dictionnary. It
    is possible to create spectrum for each catalog, they will be stored in a different
    entry.
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    Emin  : minimal energy for the plot : unit is defined by the scale (see _init__ function)
    Emax  : maximal energy for the plot : unit is defined by the scale (see _init__ function)
    '''
    if self.CatalogData[key]['found'] == False: 
        self.error("This source does not belong to "+key)
    # try:
    if 1:
      model = self.CatalogData[key]['model']
      if key == "2FHL" :
        model = "PowerLaw2"
        data = self.ReadPL2(key)
      if model=="PowerLaw":
        data = self.ReadPL(key)
      if model=="LogParabola":
        data = self.ReadLP(key)
      if model=="PLExpCutoff":
        data = self.ReadPLcutOff(key)
      if model=="PLSuperExpCutoff":
        data = self.ReadPLSupercutOff(key)
    # except :
    #   self.error("No such catalog: "+key)
    
    try :
        self.CatalogData[key]['spectrum'] = PlotLibrary.Spectrum(data,Model=model,Emin=Emin,
                                Emax=Emax,Representation=self.Representation,escale=self.escale,
                                Npt=1000)
    except :
        self.error("cannot create spectrum for model "+model+" and catalog "+key)
    self.success("Reading spectral informations from "+key+" for model "+model)


  def ReadPL(self,key):
    '''
    read the information of the catalog in the case of a Power Law model
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    indice = self.CatalogData[key]['indice']
    if key == "4FGL":
      index  = self.CatalogData[key]['data'].field('PL_Index')[indice]
      eindex = self.CatalogData[key]['data'].field('Unc_PL_Index')[indice]
      flux   = self.CatalogData[key]['data'].field('PL_Flux_Density')[indice]
      eflux  = self.CatalogData[key]['data'].field('Unc_PL_Flux_Density')[indice]  
    else :
      index  = self.CatalogData[key]['data'].field('Powerlaw_Index')[indice]
      eindex = self.CatalogData[key]['data'].field('Unc_Spectral_Index')[indice]
      flux   = self.CatalogData[key]['data'].field('Flux_Density')[indice]
      eflux  = self.CatalogData[key]['data'].field('Unc_Flux_Density')[indice]
      pivot  = self.CatalogData[key]['data'].field('Pivot_Energy')[indice]

    pivot  = self.CatalogData[key]['data'].field('Pivot_Energy')[indice]
  
    if key == '1FHL' or key == '3FHL' :
      pivot *= 1e3
      flux  *= 1e-3
      eflux *= 1e-3

    pivot = self._HandleEnergyUnit(pivot)
    flux = self._HandleFluxUnit(flux)
    eflux = self._HandleFluxUnit(eflux)
    return [flux,eflux,index,eindex,pivot]


  def ReadPLcutOff(self,key):
    '''
    read the information of the catalog in the case of a PLExpCutoff model
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    indice = self.CatalogData[key]['indice']

    if key == "4FGL":
      index  = self.CatalogData[key]['data'].field('PLEC_Index')[indice]
      eindex = self.CatalogData[key]['data'].field('Unc_PLEC_Index_Index')[indice]
    else :
      index  = self.CatalogData[key]['data'].field('Powerlaw_Index')[indice]
      eindex = self.CatalogData[key]['data'].field('Unc_Spectral_Index')[indice]

    flux   = self.CatalogData[key]['data'].field('Flux_Density')[indice]
    eflux  = self.CatalogData[key]['data'].field('Unc_Flux_Density')[indice]
    pivot  = self.CatalogData[key]['data'].field('Pivot_Energy')[indice]

    cutoff = self.CatalogData[key]['data'].field('Cutoff')[indice]
    ecutoff = self.CatalogData[key]['data'].field('Unc_Cutoff')[indice]
    
    if key == '1FHL' or key == '3FHL' :
      pivot *= 1e3
      flux  *= 1e-3
      eflux *= 1e-3

    pivot = self._HandleEnergyUnit(pivot)
    flux = self._HandleFluxUnit(flux)
    eflux = self._HandleFluxUnit(eflux)
    return [flux,eflux,index,eindex,cutoff,ecutoff,pivot]


  def ReadPLSupercutOff(self,key):
    '''
    read the information of the catalog in the case of a PLSuperExpCutoff model
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    indice = self.CatalogData[key]['indice']
    index  = self.CatalogData[key]['data'].field('Powerlaw_Index')[indice]
    eindex = self.CatalogData[key]['data'].field('Unc_Spectral_Index')[indice]

    flux   = self.CatalogData[key]['data'].field('Flux_Density')[indice]
    eflux  = self.CatalogData[key]['data'].field('Unc_Flux_Density')[indice]
    pivot  = self.CatalogData[key]['data'].field('Pivot_Energy')[indice]

    cutoff = self.CatalogData[key]['data'].field('Cutoff')[indice]
    ecutoff = self.CatalogData[key]['data'].field('Unc_Cutoff')[indice]
    
    beta = self.CatalogData[key]['data'].field('Exp_Index')[indice]
    ebeta = self.CatalogData[key]['data'].field('Unc_Exp_Index')[indice]

    if key == '1FHL' or key == '3FHL' :
      pivot *= 1e3
      flux  *= 1e-3
      eflux *= 1e-3

    pivot = self._HandleEnergyUnit(pivot)
    flux = self._HandleFluxUnit(flux)
    eflux = self._HandleFluxUnit(eflux)
    return [flux,eflux,index,eindex,cutoff,ecutoff,beta,ebeta,pivot]

  def ReadPL2(self,key):
    '''
    read the information of the catalog in the case of a Power Law 2 model
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    if key == '2FHL' :
        indice = self.CatalogData[key]['indice']
        index  = self.CatalogData[key]['data'].field('Spectral_Index')[indice]
        eindex = self.CatalogData[key]['data'].field('Unc_Spectral_Index')[indice]
        flux   = self.CatalogData[key]['data'].field('Flux50')[indice]
        eflux  = self.CatalogData[key]['data'].field('Unc_Flux50')[indice]
    else:
        self.error("No PL2 infor for catalog: "+key)
    
    return [flux,eflux,index,eindex]


  def ReadLP(self,key):
    '''
    read the information of the catalog in the case of a LogParabola model
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    indice = self.CatalogData[key]['indice']


    if key == "4FGL":
      index  = self.CatalogData[key]['data'].field('LP_Index')[indice]
      eindex = self.CatalogData[key]['data'].field('Unc_LP_Index')[indice]
      beta = self.CatalogData[key]['data'].field('LP_beta')[indice]
      ebeta = self.CatalogData[key]['data'].field('Unc_LP_beta')[indice]
      flux   = self.CatalogData[key]['data'].field('LP_Flux_Density')[indice]
      eflux  = self.CatalogData[key]['data'].field('Unc_LP_Flux_Density')[indice]
    else :
      index  = self.CatalogData[key]['data'].field('Spectral_Index')[indice]
      eindex = self.CatalogData[key]['data'].field('Unc_Spectral_Index')[indice]
      beta = self.CatalogData[key]['data'].field('beta')[indice]
      ebeta = self.CatalogData[key]['data'].field('Unc_beta')[indice]
      flux   = self.CatalogData[key]['data'].field('Flux_Density')[indice]
      eflux  = self.CatalogData[key]['data'].field('Unc_Flux_Density')[indice]
    pivot  = self.CatalogData[key]['data'].field('Pivot_Energy')[indice]
    
    if key == '1FHL'  or key == '3FHL' :
      pivot *= 1e3
      flux  *= 1e-3
      eflux *= 1e-3

    pivot = self._HandleEnergyUnit(pivot)
    flux = self._HandleFluxUnit(flux)
    eflux = self._HandleFluxUnit(eflux)
    return [flux,eflux,index,eindex,beta,ebeta,pivot]

  def Plot(self,key):
    '''
    Compute model and butterfly for the catalog key
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    if self.CatalogData[key]['found'] == False: 
        self.error("This source does not belong to "+key)
    if not('spectrum' in self.CatalogData[key]):
        self.error("No spectrum computed for "+key)

    ## Draw part
    enerbut,but = self.CatalogData[key]['spectrum'].GetButterfly()
    enerphi,phi = self.CatalogData[key]['spectrum'].GetModel()

    #enerbut = self._HandleUnit(enerbut)
    #enerphi = self._HandleUnit(enerphi)

    return enerbut,but,enerphi,phi


  def Association(self,key,asso = 'ASSOC1'):
    '''
    Look for the association of the sources based on a field given by the used; defaul is ASSOC1
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    asso  : Name of the association column in the fits file
    '''
    try:
      if self.CatalogData[key]['found'] == False: 
        self.error("This source does not belong to "+key)
      
      return self.CatalogData[key]['data'].field(asso)[self.CatalogData[key]['indice']]

    except :
      self.error("No association "+asso+" in catalog: "+key)


  def IsTeVSource(self,key):
    '''
    Look for TEV Flag in the catalog
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    Flag = "r"
    try :
      Flag = self.CatalogData[key]['data'].field('TEVCAT_FLAG')[self.CatalogData[key]['indice']]
    except  :
      self.warning("No TeV Flag in catalog: "+key)
    if  Flag == "P":
      return True
    return False


  def GetSignificance(self,key):
    '''
    Look for the average significance of the sources
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc...
    '''
    return self.CatalogData[key]['data'].field("Signif_Avg")[self.CatalogData[key]['indice']]


  def GetVarIndex(self,key):
    '''
    return the variability index of the source
    Parameters
    ----------
    key   : name of the catalog 2FGL, 3FGL, etc... 
    '''
    try : 
      if self.CatalogData[key]['found'] == False: 
        self.error("This source does not belong to "+key)
      try :
        VarI=self.CatalogData[key]['data'].field('Variability_Index')
      except:
        return -1
      return VarI[self.CatalogData[key]['indice']]

    except :
      self.error("No such catalog: "+key)


  def GetRedshift(self,key):
    '''
    return the variability index of the source
    Parameters
    ----------
    key   : name of the catalog here only 3FHL
    '''
    try : 
      if self.CatalogData[key]['found'] == False: 
        self.error("This source does not belong to "+key)
      try :
        Redshift=self.CatalogData[key]['data'].field('Redshift')
      except:
        return -1
      return Redshift[self.CatalogData[key]['indice']]
    except :
      self.error("No such catalog: "+key)

  def ReadBzCatdata(self,k):
    #Load BzCat 
    self.info("Reading BzCat ed5 data")
    BzCat = numpy.genfromtxt(self.folder+'/BzCat_ed5.csv',dtype=str,unpack=True)
    ra0, dec0 = self.GetPosition(k,"galactic")

    r = calcAngSepDeg(numpy.float32(BzCat[8]),numpy.float32(BzCat[9]),ra0,dec0)
      # print res
    res = r.argmin()
    if min(r) < 0.4:
      return BzCat[1][res],float(BzCat[10][res]),float(BzCat[11][res]),float(BzCat[13][res]),float(BzCat[14][res]),float(BzCat[15][res])
  #     self.warning("No source closer than 0.5 degrees in catalog ",k)
  #     self.CatalogData[k]['found'] = False

  #     return
  #   self.success("found a close source in the catalog "+self.CatalogData[k]['fits']+" at r="+str(r[res])+" named "+self.CatalogData[k]['data'].field('Source_Name')[res])
  #   self.CatalogData[k]['indice'] = res
  #   self.CatalogData[k]['name'] = self.CatalogData[k]['data'].field('Source_Name')[res]
  #   self.CatalogData[k]['found'] = True

    return "Null",-1,0,0,0,0


  def Read3lacdata(self,k):
    #Load 3LAC 
    self.info("Reading 3 LAC data")
    ThreeLacCat = numpy.genfromtxt(self.folder+'/3LAC_list.dat',dtype=str,unpack=True)
    ra0, dec0 = self.GetPosition(k,"galactic")
    r = calcAngSepDeg(numpy.float32(ThreeLacCat[10]),numpy.float32(ThreeLacCat[11]),ra0,dec0)
    res = r.argmin()
    if min(r) < 0.4:
      return ThreeLacCat[1][res]+"_"+ThreeLacCat[2][res],float(ThreeLacCat[12][res]),float(ThreeLacCat[14][res]),float(ThreeLacCat[15][res])

    return "Null",-1,0,0

  def GetRedshift4LAC(self):
    # Load 4LAC
    self.info("Reading 4 LAC data")
    table = fits.open(self.folder+"/table_4LAC.fits")[1].data  
    Name = self.GetCatalogName("4FGL") 
    for i in range(len(table)): 
      if table[i][0]  == Name :
        return  table[i]["Redshift"]  
    return -1

  def GetSyncPeak4LAC(self):
    # Load 4LAC
    self.info("Reading 4 LAC data")
    table = fits.open(self.folder+"/table_4LAC.fits")[1].data  
    Name = self.GetCatalogName("4FGL") 
    for i in range(len(table)): 
      if table[i][0]  == Name :
        return  table[i]["nu_syn"]  
    return -1

  def GetSyncPeak3LAC(self,k="3FGL"):
    # Load 4LAC
    self.info("Reading 3 LAC data")
    ThreeLacCat = numpy.genfromtxt(self.folder+'/3LAC_list.dat',dtype=str,unpack=True)
    Name = self.GetCatalogName("4FGL") 
    ra0, dec0 = self.GetPosition(k,"galactic")
    r = calcAngSepDeg(numpy.float32(ThreeLacCat[10]),numpy.float32(ThreeLacCat[11]),ra0,dec0)
    res = r.argmin()
    if min(r) < 0.4:
      return ThreeLacCat[-1][res]
    return -1

  def _HandleEnergyUnit(self,ener):
      fac = 1
      if self.escale == "TeV":
          fac = 1e-6
      if self.escale == "GeV":
          fac = 1e-3
      return ener*fac

  def _HandleFluxUnit(self,flux):
      fac = 1
      if self.escale == "TeV":
          fac = 1e6
      if self.escale == "GeV":
          fac = 1e3
      return fac*flux



#Side functions
def calcAngSepDeg(ra0, dec0, ra1, dec1):
    '''Return the angular separation between two objects. Use the
    special case of the Vincenty formula that is accurate for all
    distances'''
    C = numpy.pi / 180
    d0 = C * dec0
    d1 = C * dec1
    r12 = C * (ra0 - ra1)
    cd0 = numpy.cos(d0)
    sd0 = numpy.sin(d0)
    cd1 = numpy.cos(d1)
    sd1 = numpy.sin(d1)
    cr12 = numpy.cos(r12)
    sr12 = numpy.sin(r12)
    num = numpy.sqrt((cd0 * sr12) ** 2 + (cd1 * sd0 - sd1 * cd0 * cr12) ** 2)
    den = sd0 * sd1 + cd0 * cd1 * cr12
    return numpy.arctan2(num, den) / C



