"""
Example script to run the GRBModelling class
on the data from the GRB190829A detected by the H.E.S.S. Collaboration
using the module written specifically for this GRB.

Author: Carlo Romoli (MPIK)
"""

import numpy as np
import matplotlib.pyplot as plt

from astropy.table import Table, hstack
import astropy.units as u
from astropy.io import ascii
import naima

from grbloader_forhess import *

# tab = ascii.read("magic_int1_points.txt")
# newt = Table([tab['energy'],tab['flux'],tab['flux_hi']-tab['flux'],tab['flux']-tab['flux_lo']],
#              names=['energy','flux','flux_error_hi','flux_error_lo'])

n1_xray_tab = Table(ascii.read("./grb190829a_n1_modeldata/xtable_n1_ipac.txt"))
n1_lat_tab = Table(ascii.read("./grb190829a_n1_modeldata/lat_n1_ipac.txt"))
n1_hess_tab = Table(ascii.read("./grb190829a_n1_modeldata/hess_n1_ipac.txt"))

n2_xray_tab = Table(ascii.read("./grb190829a_n2_modeldata/xray_n2_ipac.txt"))
n2_hess_tab = Table(ascii.read("./grb190829a_n2_modeldata/hess_n2_ipac.txt"))

n1_dataset = [n1_xray_tab, n1_lat_tab, n1_hess_tab]
n2_dataset = [n2_xray_tab, n2_hess_tab]

Eiso = 2e50   # erg
density = 1.  # cm-3
redshift = 0.0785
tstart = 15633   # s
tstop = 27746    # s

tstart2 = 97855
tstop2 = 115207

# Standard type of initialization
grb190829a = GRBModelling(Eiso, density, n1_dataset, tstart, tstop, 0.0785,
                          [-0.04, -1.7, 3.1, 1.6, -0.45],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)'],
                          scenario='ISM',
                          free_doppler=False,
                          cooling_constrain=True,
                          synch_nolimit=False
                          )

testrun = grb190829a.run_naima("./grb190829a_n1_ism_ssc", 128, 100, 200, 2, prefit=False)


pars = [np.median(a) for a in testrun.flatchain.T]
grb190829a.pars = pars

newene = Table([np.logspace(-1, 13, 500) * u.eV], names=['energy'])
naima.plot_data(grb190829a.dataset)
model = grb190829a.naimamodel(grb190829a.pars, newene)[0]
plt.loglog(newene, model, 'k-', label="TOT", lw=3, alpha=0.3)
plt.loglog(newene, grb190829a.synch_comp, 'k--', alpha=0.5, label="Synch.")
plt.loglog(newene, grb190829a.ic_comp, 'k-.', alpha=0.5, label="IC no abs.")
plt.loglog(newene, grb190829a.ic_compGG, 'k:', alpha=0.5, label="IC abs. method 1")

newene = [1e-4*u.eV, 1e13*u.eV]
a, b = naima.plot._calc_CI(testrun, confs=[1], modelidx=0, e_range=newene)
xval = a.value
ymax1 = b[0][1].value  # 1 sigma
ymin1 = b[0][0].value  # 1 sigma
plt.fill_between(xval,
                 ymax1,
                 ymin1,
                 alpha=0.2,
                 color='C0',
                 label="1$\sigma$")

plt.ylim(0.9e-12, 1.1e-10)
plt.xlim(1.e-2, 1e13)
plt.xlabel("Energy [eV]", size=13)
plt.ylabel("$E^2\mathrm{d}N/\mathrm{d}E$ [$\mathrm{erg\,s^{-1}\,cm^{-2}}$]", size=13)
ax = plt.gca()
ax.tick_params(axis='both', labelsize=12)
plt.legend()

plt.show()

"""
In the article different cases have been tested. To specifically rerun those use the following starting points
NOTE: MCMC have an element of stochasticity so if you rerun the fits, the results might not be exactly the same
as those published, but anyway compatible.

1- Night 1 SSC scenario

grb190829a = GRBModelling(Eiso, density, n1_dataset, tstart, tstop, 0.0785,
                          [-0.04, -1.7, 3.1, 1.6, -0.45],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)'],
                          scenario='ISM',
                          cooling_constrain=True,
                          synch_nolimit=False
                          )
testrun = grb190829a.run_naima("./grb190829a_n1_ism_ssc", 128, 100, 200, 2, prefit=False)


2- Night 1 SSC without limit scenario
grb190829a = GRBModelling(Eiso, density, n1_dataset, tstart, tstop, 0.0785,
                          [-1.6, -2.5, 3.1, 4.0, 0.2],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)'],
                          scenario='ISM',
                          cooling_constrain=False,
                          synch_nolimit=True
                          )
testrun = grb190829a.run_naima("./grb190829a_n1_ism_syn", 128, 100, 200, 2, prefit=False)

3- Night 2 SSC scenario
grb190829a = GRBModelling(Eiso, density, n2_dataset, tstart2, tstop2, 0.0785,
                          [-0.04, -1.5, 3.1, 1.6, -0.6],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)'],
                          scenario='ISM',
                          cooling_constrain=True,
                          synch_nolimit=False
                          )
testrun = grb190829a.run_naima("./grb190829a_n2_ism_ssc", 128, 100, 200, 2, prefit=False)

4- Night 2 SSC without limit scenario
grb190829a = GRBModelling(Eiso, density, n2_dataset, tstart2, tstop2, 0.0785,
                          [-1.1, -2.5, 3.1, 4.0, 0.1],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)'],
                          scenario='ISM',
                          cooling_constrain=False,
                          synch_nolimit=True
                          )
testrun = grb190829a.run_naima("./grb190829a_n2_ism_syn", 128, 100, 200, 2, prefit=False)

5- Night 1 SSC scenario with free Doppler boosting
grb190829dopp = GRBModelling(Eiso, density, n1_dataset, tstart, tstop, 0.0785,
                          [-0.5, -2.8, 3.0, 1.3, -1.7, 2.5],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)', 'log10(Dopp)'],
                           scenario='ISM',
                          free_doppler=True,
                          cooling_constrain=False,
                          synch_nolimit=False
                          )
testrun = grb190829dopp.run_naima("./grb190829a_n1_ism_ssc_dopp", 128, 100, 200, 2, prefit=False)

6- Night 1 SSC scenario with added optical database
# For this one the user needs to load also the optical data
# It also requires modifications to all the prior function so that the
# limit on the break refers still to the X-rays (that now are dataset[1] instead of dataset[0])
# the modification needs to be at line 376 (change from self.dataset[0] to self.dataset[1])

n1_xray_tab = Table(ascii.read("./grb190829a_n1_modeldata/test_xtable_n1_ipac.txt"))
n1_lat_tab = Table(ascii.read("./grb190829a_n1_modeldata/test_lat_n1_ipac.txt"))
n1_hess_tab = Table(ascii.read("./grb190829a_n1_modeldata/test_hess_n1_ipac.txt"))
n1_optical = Table(ascii.read("./grb190829a_n1_modeldata/gtc_optical_ene_n1.txt"))  # not used, only for checking
n1_dataset_opt = [n1_optical, n1_xray_tab, n1_lat_tab, n1_hess_tab]

grb190829a = GRBModelling(Eiso, density, n1_dataset_opt, tstart, tstop, 0.0785,
                          [-0.04, -1.7, 3.1, 1.6, -0.45],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)'],
                          scenario='ISM',
                          cooling_constrain=True,
                          synch_nolimit=False
                          )
testrun = grb190829a.run_naima("./grb190829a_n1_ism_ssc_opt", 128, 100, 200, 2, prefit=False)


7- Night 1 SSC without limit scenario with added optical data
# For this one the user needs to load also the optical data
# It also requires modifications to all the prior function so that the
# limit on the break refers still to the X-rays (that now are dataset[1] instead of dataset[0])
# the modification needs to be at line 376 (change from self.dataset[0] to self.dataset[1])

n1_xray_tab = Table(ascii.read("./grb190829a_n1_modeldata/test_xtable_n1_ipac.txt"))
n1_lat_tab = Table(ascii.read("./grb190829a_n1_modeldata/test_lat_n1_ipac.txt"))
n1_hess_tab = Table(ascii.read("./grb190829a_n1_modeldata/test_hess_n1_ipac.txt"))
n1_optical = Table(ascii.read("./grb190829a_n1_modeldata/gtc_optical_ene_n1.txt"))  # not used, only for checking
n1_dataset_opt = [n1_optical, n1_xray_tab, n1_lat_tab, n1_hess_tab]

grb190829a = GRBModelling(Eiso, density, n1_dataset_opt, tstart, tstop, 0.0785,
                          [-1.6, -2.5, 3.1, 4.0, 0.2],
                          ['log10(eta_e)', 'log10(Ebreak)', 'Index2', 'log10(Ec)', 'log10(B)'],
                          scenario='ISM',
                          cooling_constrain=False,
                          synch_nolimit=True
                          )
testrun = grb190829a.run_naima("./grb190829a_n1_ism_syn_opt", 128, 100, 200, 2, prefit=False)
"""