# GRB modelling with NAIMA
#
# Author: C. Romoli - MPIK
#
# Thanks to F. Aharonian, A. Taylor, D. Khangulyan
# that helped with the theoretical framework
#
# Class and functions to model GRB using the NAIMA package.
#
# References:
# - HESS Collaboration, 2020 - paper on the GRB190829A VHE emission. - work in progress
# - Eungwanichayapant, A. & Aharonian, F., 2009 https://ui.adsabs.harvard.edu/abs/2009IJMPD..18..911E/abstract
# - Aharonian, 2004 - https://ui.adsabs.harvard.edu/abs/2004vhec.book.....A/abstract
# - Aharonian, 2000 - https://ui.adsabs.harvard.edu/abs/2000NewA....5..377A/abstract
# - Atoyan & Aharonian, 1996 - http://adsabs.harvard.edu/abs/1996MNRAS.278..525A
# - Rybicki & Lightman, 1979 - https://ui.adsabs.harvard.edu/abs/1979rpa..book.....R/abstract
#
# In this particular branch, the code is optimized for the use on the data of the
# GRB 190827A
#
import numpy as np

import astropy
from astropy.table import Table
import astropy.units as u
import astropy.constants as con
from astropy.cosmology import WMAP9 as cosmo

from scipy.integrate import quad as integ
from astropy.time import Time

import naima
from naima.models import Synchrotron, InverseCompton, ExponentialCutoffBrokenPowerLaw, PowerLaw
from naima import uniform_prior, normal_prior

import matplotlib.pyplot as plt
# static variables:
#
m_e = con.m_e.cgs.value
c = con.c.cgs.value
mec2_eV = (con.m_e * con.c ** 2.).to('eV').value
h = con.h.cgs.value
el = con.e.gauss.value
erg_to_eV = 624150912588.3258  # conversion from erg to eV
sigma_T = con.sigma_T.cgs.value
mpc2 = (con.m_p * con.c ** 2.).to('eV')
mpc2_erg = mpc2.to('erg').value

"""
Bunch of auxiliary functions needed by the loaddataset_gammapy() function
"""
def fitfunc(x,n,a,s):
    """
    auxiliary function PL
    """
    return n*(x/s)**(a)

def fitfunc_N(x,s,a):
    """
    derivative wrt N of auxiliary function PL
    """
    return (x/s)**(a)

def fitfunc_a(x,n,a,s):
    """
    derivative wrt a of auxiliary function PL
    """
    return fitfunc(x,n,a,s)*np.log(x/s)*np.sign(a)


# static methods:
#
# Functions for calculation of gamma-gamma absorption

def sigma_gammagamma(Eph1, Eph2):
    """
    gamma-gamma cross section averaged over scattering angle
    The value is returned in cm2

    Equation 5) from Eungwanichayapant, A.; Aharonian, F., 2009
    https://ui.adsabs.harvard.edu/abs/2009IJMPD..18..911E/abstract
    (originally from Aharonian, 2004 - https://ui.adsabs.harvard.edu/abs/2004vhec.book.....A/abstract)
    Approximation good within 3%

    Parameters
    ----------
       Eph1 : array_like
         numpy array of energy of gamma ray in eV
       Eph2 : array_like
         np.array of energy of target photon in eV
    Returns
    -------
        cross_section : astropy.quantity
          angle average cross section for gamma gamma absorption
    """

    CMene = Eph1 * Eph2 / (mec2_eV * mec2_eV)
    mask = CMene > 1.  # mask condition to account for the threshold effect.
    res = np.full(CMene.shape, 0.)
    res[mask] = 3. / (2. * CMene[mask] * CMene[mask]) * sigma_T * \
                     ((CMene[mask] + 0.5 * np.log(CMene[mask]) - 1. / 6. + 1. / (2. * CMene[mask]))
                         * np.log(np.sqrt(CMene[mask]) + np.sqrt(CMene[mask] - 1)) -
                         (CMene[mask] + 4. / 9. - 1. / (9. * CMene[mask])) *
                         np.sqrt(1. - (1. / CMene[mask])))
    cross_section = res * u.cm * u.cm
    return cross_section


def absorption_coeff(egamma, targetene, target):
    """
    Returns the absorption coefficient K that will then be
    spatially integrated.

    K(E) = \int_e sigma_gg(E,e) * dn/de * de

    where E is the gamma ray energy, e is the energy of the target photon and dn/de is the number distribution
    of the target photon field. (Inner integral of equation 3.24 of
    Aharonian, 2004 - https://ui.adsabs.harvard.edu/abs/2004vhec.book.....A/abstract)

    Parameters
    ----------
      egamma : array_like
        Energy of the gamma ray photon as astropy unit quantity. Format e.g. [1.]*u.TeV
      targetene : array_like
        energy array of the photon distribution e.g. [1.,2.]*u.eV
      target : array_like
        dnde value of the target photon distribution as 1/(eV cm3)
    Returns
    -------
      abs_coeff : astropy.quantity
        absorption coefficient as astropy quantity (should have units 1/u.cm)
    """

    product = sigma_gammagamma(np.vstack(egamma.to('eV')),
                               targetene.to('eV').value) * target  # make sure the units are correct
    abs_coeff = naima.utils.trapz_loglog(product, targetene, axis=1)
    return abs_coeff


def tau_val(Egamma, targetene, target, size):
    """
    Optical depth assuming homogeneous radiation field

    From equation 3.24 of Aharonian, 2004 - https://ui.adsabs.harvard.edu/abs/2004vhec.book.....A/abstract
    with the assumption of homogeneous photon field.

    Parameters
    ----------
      Egamma    : array_like
        Energy of the gamma ray photon as astropy unit quantity. Format e.g. [1.]*u.TeV
      targetene : array_like
        energy array of the photon distribution e.g. [1.,2.]*u.eV
      target    : array_like
        dnde value of the target photon distribution as 1/(eV cm3)
      size      : astropy.quantity
        size of the integration length as astropy spatial quantity (normally units of cm)
    Returns
    -------
      tau : array_like
        optical depth
    """

    coeff = absorption_coeff(Egamma, targetene, target)
    tau = size.to('cm') * coeff
    return tau


def cutoff_limit(bfield):
    """
     Account for the maximum energy of particles
     for synchrotron emission. Due to the effect of the synchrotron burn-off
     due to the balancing of acceleration and losses of the particle in magnetic field.
     Expression 18 from Aharonian, 2000

    Parameters
    ----------
      bfield : float
        Magnetic field intensity to be given in units of Gauss
    Returns
    -------
      cutoff_ev : float
        log10 of the cutoff energy in units of TeV
    """

    eff = 1.  # acceleration efficiency parameter (eff >= 1)
    cutoff = ((3. / 2.) ** (3. / 4.) * np.sqrt(1. / (el ** 3 * bfield)) * (m_e ** 2. * c ** 4.)) * eff ** (-0.5) * u.erg
    cutoff_TeV = (cutoff.value * erg_to_eV * 1e-12)
    return np.log10(cutoff_TeV)


def synch_cooltime_partene(bfield, partene):
    """
    Computes the cooling time for an electron with energy 'partene' in
    Bfield. Returns in units of seconds
    Equation 1 from Aharonian, 2000

    Parameters
    ----------
       bfield : astropy.quantity
         magnetic field as astropy quantity (u.G)
       partene : astropy.quantity
         particle energy as astropy quantity (u.eV)
    Returns
    -------
       tcool : astropy.quantity
         Synchrotron cooling time as astropy quantity (u.s)
    """

    bf = bfield.to('G').value
    epar = partene.to('erg').value
    tcool = (6. * np.pi * m_e ** 4. * c ** 3.) / (sigma_T * m_e ** 2. * epar * bf ** 2.)
    return tcool * u.s


def synch_charene(bfield, partene):
    """
    Function to return
    characteristic energy of synchrotron spectrum

    Equation 3.30 from Aharonian, 2004 (adapted for electrons)

    Parameters
    ----------
       bfield : astropy.quantity
         magnetic field as astropy quantity (u.G)
       partene : astropy.quantity
         particle energy as astropy quantity (u.eV)
    Returns
    -------
       charene : astropy.quantity
         synchrotron characteristic energy as astropy quantity (u.eV)
    """

    bf = bfield.to('G').value
    epar = partene.to('erg').value
    charene = np.sqrt(3. / 2.) * (h * el * bf) / \
                     (2. * np.pi * (m_e ** 3. * c ** 5.)) * epar ** 2.  # in ergs
    return charene * erg_to_eV * u.eV


class ArgumentConflict(Exception):
    """
    Custom Exception to be raised when arguments in the initialization
    of the modelling class conflict.
    """
    pass


class GRBModelling:
    """
    Class to produce the grb modelling. The spectral modelling presented here
    is based on the picture of particle acceleration at the forward shock,
    which propagates outwards through the circumburst material
    (see   `R. D. Blandford, C. F. McKee,Physics of Fluids19, 1130 (1976)`).
    Given the total isotropic energy of the explosion
    (`Eiso`), the density of material surrounding the GRB (`n`) and the time of the observation (after trigger),
    it computes the physical parameters of the GRB, like the Lorentz factor `gamma` and the size of the emitting
    shell.

    This class has been used to model the multiwavelength emission of the H.E.S.S. GRB `GRB190829A`.

    Attributes
    ----------
    Eiso : float
        Isotropic energy of the GRB (in units of erg)
    density : float
        density of the circumburst material (in units of cm-3)
    dataset : list of astropy.table.table.Table
        table of observational data. Attribute exists only if a list of tables is passed in the initialization
    tstart : float
        starting time of the observational interval (in units of seconds)
    tstop : float
        stop time of the observational interval (in units of seconds)
    avtime : float
        average time of the observational interval
    redshift : float
        redshift of the GRB
    Dl : astropy.quantity
        luminosity distance of the GRB (as astropy quantity)
    pars : list
        list of parameters of a naima.models.ExponentialCutoffBrokenPowerLaw
    labels : list
        list of parameter names (as strings)
    cooling_constrain : boolean
        If True adds to the prior a constrain for which cooling time at break ~ age of the system. DEFAULT = True
        If synch_nolimit = True, this option does not do anything.
    synch_nolimit : boolean
        False for standard SSC model, True for synchtrotron dominated model. DEFAULT = False
    gamma : float
        Lorentz factor of the GRB at time avtime
    sizer : float
        radius of the expanding shell at time avtime
    shock_energy : astropy.quantity (u.erg)
        available energy in the shock
    Emin : astropy.quantity
        minimum injection energy of the electron distribution
    Wesyn : astropy.quantity
        total energy in the electron distribution
    eta_e : float
        fraction of available energy ending in the electron distribution
    eta_b : float
        fraction of available energy ending in magnetic field energy density
    synch_comp : numpy.array
        synchrotron component of the emission
    ic_comp : numpy.array
        inverse compton component of the emission
    synch_compGG : numpy.array
        synchrotron component of the emission
        with gamma gamma absorption included METHOD 1
    ic_compGG : numpy.array
        inverse compton component of the emission
        with gammagamma absorption included METHOD 1
    synch_compGG2 : numpy.array
        synchrotron component of the emission
        with gamma gamma absorption included METHOD 2
    ic_compGG2 : numpy.array
        inverse compton component of the emission
        with gammagamma absorption included METHOD 2
    naimamodel : bound method
        bound method to the model function
        associated with function load_model_and_prior()
    lnprior : bound method
        bound method to the prior function
        associated with function load_model_and_prior()
    """

    def __init__(self, eiso, dens, data, tstart, tstop, redshift, pars, labels,
                 scenario='ISM',
                 mass_loss=0,
                 wind_speed=0,
                 cooling_constrain=True,
                 synch_nolimit=False,
                 free_doppler=False,
                 all_free=False,
                 electron_bins=False):
        """
        Class initialization

        Parameters
        ----------
          eiso : float
            Isotropic energy of the gamma ray burst (given in erg)
          dens : float
            density of the circumburst material (given in cm-3) valid only for average and ISM scenario
          data : list
            list of astropy table with the obs. data. Optional, theoretical line can be computed anyway
          tstart : float
            start time of the observational interval (given in seconds after trigger)
          tstop : float
            stop time of the observational interval (given in seconds after trigger)
          redshift : float
            redshift of the GRB
          pars : list
            list of parameters passed to the model function
          labels : list
            names of the parameters passed to the model
          mass_loss : float
            value of the mass loss rate of the progenitor (given in solar masses per year)
          wind_speed : float
            value of the speed of the stellar wind of the projenitor (given in km/s)
          cooling_constrain : bool
            boolean to add a contrain on cooling time at break ~ age of of the system in the prior function
          synch_nolimit : bool
            boolean to select the synchrotron dominated model
          free_doppler : bool
            to choose the model with a free doppler factor
          all_free : bool
            to choose the model with all parameters of the broken powerlaw set free
          electron_bins : bool
            to choose the model with an electron distribution made of power law segments
        """

        if isinstance(data, list):
            if all(isinstance(x, astropy.table.table.Table) for x in data):
                self.dataset = data  # dataset astropy table
                self.eblimdata = self.dataset[0]  # TEMPORARY, assigns directly the break limit to specific dataset
            else:
                print("WARNING: Not all the elements in your list are formatted as astropy tables!")
                print("Not loading the dataset,")
                print("the code can be used only for computation of theoretical curves")
        else:
            print("WARNING: No dataset given,")
            print("the code can be used only for computation of theoretical curves")
        self.Eiso = eiso  # Eiso of the burst
        self.density = dens  # ambient density around the burst units of cm-3
        self.mass_loss = mass_loss  # Value of the mass loss rate of progenitor in solar masses per year
        self.wind_speed = wind_speed  # Value of the wind speed of the projenitor in km/s
        self.tstart = tstart  # units of s
        self.tstop = tstop  # units of s
        self.avtime = (tstart + tstop) / 2.  # units of s
        self.redshift = redshift
        self.Dl = cosmo.luminosity_distance(redshift)  # luminosity distance with units
        self.pars = pars  # parameters for the fit
        self.labels = labels  # labels of the parameters
        self.scenario = scenario  # string valid options: 'average', 'Wind', 'ISM'
        self.cooling_constrain = cooling_constrain  # if True add in the prior a constrain on cooling break
        self.synch_nolimit = synch_nolimit  # boolean for SSC (=0) or synchrotron without cut-off limit model
        self.free_doppler = free_doppler  # boolean for having the doppler factor as a free parameter
        self.all_free = all_free  # boolean for having all the parameters of the broken power law left free
        self.electron_bins = electron_bins  # boolean for having the electron distribution as PL segments
        self.gamma = 0  # Gamma factor of the GRB at certain time
        self.sizer = 0  # External radius of the shell
        self.shock_energy = 0  # Available energy in the shock
        self.Emin = 0 * u.eV  # Minimum injection energy for the particle distribution
        self.Wesyn = 0  # Total energy in the electrons
        self.eta_e = 0  # Fraction of thermal energy going into electron energy
        self.eta_b = 0  # Fraction of thermal energy going into magnetic field
        self.synch_comp = 0  # Spectrum of the synchrotron component
        self.ic_comp = 0  # Spectrum of the IC component
        self.naimamodel = 0  # Model used for the fit - initialized in later function
        self.lnprior = 0  # Prior used for the fit - initialized in later function
        self.depthpar = 0  # private attribute to control the depth of the shock: d = R/(self.depthpar * Gamma)
        self.load_model_and_prior()  # Loads the NAIMA model and the relative prior
        self.esycool = 0  # Characteristic synchrotron energy corresponding to the break energy of the electrons
        self.synchedens = 0  # total energy density of synchrotron photons
        self.synch_compGG = 0  # synchrotron component of the model with gammagamma absorption with METHOD 1
        self.ic_compGG = 0  # inverse compton component of the model with gammagamma absorption with METHOD 1
        self.synch_compGG2 = 0  # synchrotron component of the model with gammagamma absorption with METHOD 2
        self.ic_compGG2 = 0  # inverse compton component of the model with gammagamma absorption with METHOD 2

    def _density_value(self):
        """
        Computes the density of the medium in the Wind scenario assuming
        a mass loss rate of the progenitor of mass_loss in solar masses per year and
        a wind speed wind_speed in km/s
        """
        if self.scenario == 'Wind':
            self.density = (self.mass_loss * 1.2e57 / 3.15e7) / (4. * np.pi * self.wind_speed * 1e5 * self.sizer**2)

    def gammaval(self):
        """
        Computes the Lorentz factor and the size of the region
        Expression from Blandford&McKee,1976.

        Gamma^2 = 3 / (4 * pi) * E_iso / (n * m_p * c^2 * R^3)

        The calculation of the radius uses the relation

        R = A * Gamma^2 * (ct)

        where A can be 4 (for Wind scenario), 8 (ISM scenario), 6 (for the average)

        Time is the average between the tstart and tstop.
        The functions takes automatically the initialization parameters
        """
        if (self.scenario == 'average'):
            self.gamma = (1. / 6.) ** (3. / 8.) * (
                3.0 * self.Eiso / (4.0 * np.pi * self.density * mpc2_erg * ((c * self.avtime) ** 3.0))) ** 0.125
            self.sizer = 6. * c * self.avtime * self.gamma ** 2.
            self.depthpar = 9. / 2.
        elif (self.scenario == 'Wind'):
            if self.mass_loss == 0 or self.wind_speed == 0:
                text = "Need to define non 0 values for the mass loss rate and the wind speed!"
                raise ValueError(text)
            self.gamma = ( ( 3. * self.Eiso * self.wind_speed * 1e5) / (4. * c**3 * self.avtime * self.mass_loss * 2e33 / 3.15e7) )**0.25
            self.sizer = 4. * self.gamma ** 2 * c * self.avtime
            self.depthpar = 3. / 1.
            self._density_value()
        elif (self.scenario == 'ISM'):
            self.gamma = (1. / 8.) ** (3. / 8.) * (
                    3.0 * self.Eiso / (4.0 * np.pi * self.density * mpc2_erg * ((c * self.avtime) ** 3.0))) ** 0.125
            self.sizer = 8. * c * self.avtime * self.gamma ** 2.
            self.depthpar = 9. / 1.
        else:
            text = "Chosen scenario: %s\n" \
                   "The scenario indicated not found. Please choose between\n" \
                   " - 'average' : average between wind and ISM scenario\n" \
                   " - 'Wind' : wind scenario\n" \
                   " - 'ISM' : ISM scenario"%self.scenario
            raise ValueError(text)

    def load_model_and_prior(self):
        """
        Associates the bound methods
        naimamodel and lnprior to
        model and prior function chosen in the initialization.

        Modify here if you want to change the model
        or the priors
        """

        self.gammaval()
        try:
            if self.free_doppler and self.all_free:
                raise ArgumentConflict
        except ArgumentConflict:
            if len(self.pars) == 7:
                print("You are giving 7 free parameters. Probably you want the all_free=True and free_doppler=False")
                print("Setting it")
                self.all_free = True
                self.free_doppler = False
            elif len(self.pars) == 6:
                print("You are giving 6 free parameters. Probably you want the all_free=False and free_doppler=True")
                print("Setting it")
                self.all_free = False
                self.free_doppler = True
        if self.free_doppler:
            self.naimamodel = self._naimamodel_free_doppler
            self.lnprior = self._lnprior_ind2free_wlim_dopp  # with free doppler, remove cooling time constrain
        elif self.all_free:
            self.naimamodel = self._naimamodel_all_free
            # with all free, the cooling time constrain is already implemented
            if self.cooling_constrain:
                self.lnprior = self._lnprior_ind2free_wlim_allfree_wcooling_constrain
            else:
                self.lnprior = self._lnprior_ind2free_wlim_allfree
        else:
            self.naimamodel = self._naimamodel_ind1fixed
            if self.synch_nolimit:
                self.lnprior = self._lnprior_ind2free_nolim
            else:
                if self.cooling_constrain:
                    self.lnprior = self._lnprior_ind2free_wlim_wcooling_constrain
                else:
                    self.lnprior = self._lnprior_ind2free_wlim
        if self.electron_bins:
            # having this option as true overwrites all the other settings,
            # use with care
            self.naimamodel = self._naima_free_electrons
            self.lnprior = self._lnprior_electron_test

    def calc_photon_density(self, Lsy, sizereg):
        """
        This is a thin shell, we use the approximation
        that the radiation is emitted in a region with radius sizereg.
        No correction factor needed because of thin shell.

        Parameters
        ----------
            Lsy : array_like
              emitted photons per second (units of 1/s)
            sizereg : astropy.quantiy
              size of the region as astropy u.cm quantity

        Returns
        -------
          ph_dens : array_like
            Photon density in the considered emission region.
        """

        # uses a sphere approximation but without the correction factor needed for a
        # full sphere (see e.g. Atoyan, Aharonian, 1996)
        # because we are in a thin shell so: n_ph = Lsy / (4 * pi * R^2 * c)
        return Lsy / (
                4. * np.pi * sizereg ** 2. * c * u.cm / u.s)

    def _naima_free_electrons(self, pars, data):
        """
        Simple function to use a very general electron distribution that would be somehow able to fit the grb data
        Constituted of 6 segments (to limit the number of free parameters) in an energy range from 1 to 1e5 GeV
        in logarithmic space.
        For this the magnetic field has been fixed to 10**-0.46 Gauss (the best solution of the standard SSC case).

        WARNING: This function has not been fully optimised. The minimum and maximum energy and the binning
        for the electron distribution might not be optimal values and are hard-coded in the model.

        Parameters
        ----------
           pars : list
             parameters of the model as list
           data : astropy.table.table.Table
             observational dataset (as astropy table) or
             if interested only in theoretical lines, astropy table
             containing only a column of energy values.
        Returns
        -------
           model : array_like
             values of the model in correspondence of the data
           electron_distribution : tuple
             electron distribution as tuple energy, electron_distribution(energy) in units of erg.
             Each element of the tuple is a list of 6 elements, one for each component of the electron spectrum.
        """
        doppler = self.gamma
        size_reg = self.sizer * u.cm
        redf = 1. + self.redshift
        bfield = 10 ** -0.46 * u.G  # fixing the magnetic field to avoid too much degeneracy in the model. HARD CODED
        emin = 1. * u.GeV  # minimum energy. HARD CODED
        emax = 1e5 * u.GeV  # maximum energy. HARD CODED
        # boundaries of the electrons energy bins
        self.ebins = ebins = np.logspace(np.log10(emin.value), np.log10(emax.value), 7) * u.GeV  # 6 bins. HARD CODED
        self.emeans = emeans = np.sqrt(ebins[1:] * ebins[:-1])  # mean energy of each electron segment
        ampl_list = []
        PL_list = []
        SYN_list = []
        Esy_list = []
        phn_sy_list = []
        tauval = 0
        SYN_model = 0
        IC_model = 0
        # hidden attributes valid only for this model. Will return the SYN spectrum for each electron bin
        self.SYN_model_list = []
        # hidden attributes valid only for this model. Will return the IC spectrum for each electron bin
        self.IC_model_list = []
        for i in range(len(ebins)-1):
            # Calculate the Synchrotron emission for each of the bins of the electron spectrum
            ampl_list.append(10 ** pars[i] * 1. / u.eV)
            PL_list.append(PowerLaw(ampl_list[i], e_0=emeans[i], alpha=3.))  # all components have -3 spectral index
            SYN = Synchrotron(PL_list[i], B=bfield, Eemin=ebins[i], Eemax=ebins[i+1], nEed=10)
            SYN_list.append(SYN)
            cutoff_charene = np.log10((synch_charene(bfield, ebins[i+1])).value)
            bins = 20
            Esy = np.logspace(-4 + np.log10(ebins[0].value), cutoff_charene + 1, bins) * u.eV
            Esy_list.append(Esy)
            # number of synchrotron photons per energy per time (units of 1/eV/s)
            Lsy = SYN_list[i].flux(Esy_list[i], distance=0. * u.cm)
            phn_sy = self.calc_photon_density(Lsy, size_reg)
            phn_sy_list.append(phn_sy)  # number density of synchrotron photons (dn/dE) units of 1/eV/cm3
            # The size of the emitting shell is given as R/(9*Gamma). This comes from Eq. 7 in HESS GRB190829A paper
            tauval += (tau_val(data['energy'] / doppler * redf, Esy, phn_sy, self.sizer / (self.depthpar * self.gamma) * u.cm))
            val = ((doppler ** 2.) * SYN.sed(data['energy'] / doppler * redf, distance=self.Dl))
            SYN_model += val
            self.SYN_model_list.append(val)  # store each synchrotron spectrum separately
        # Double loop to compute the IC emission for every electron and for every bit of synchrotron
        # radiation produced by every electron bin
        # Also store the electron distribution list
        en_range_list = []
        el_dist_list = []
        for i in range(len(ampl_list)):
            model_temp = 0
            for j in range(len(Esy_list)):
                IC = InverseCompton(PL_list[i], seed_photon_fields=[['SSC', Esy_list[j], phn_sy_list[j]]],
                                    Eemin=ebins[i], Eemax=ebins[i+1], nEed=30)
                ic_comp_temp = (doppler ** 2.) * IC.sed(data['energy'] / doppler * redf, distance=self.Dl)
                model_temp += ic_comp_temp
                self.IC_model_list.append(ic_comp_temp)  # append every IC component to the list (36 components).
            IC_model += model_temp
            en_range = np.logspace(np.log10(ebins[i].to('GeV').value), np.log10(ebins[i+1].to('GeV').value), 15) * u.GeV
            en_range_list.append(en_range)
            el_dist_list.append(PL_list[i](en_range))
        # internal absorption applied at the end on the sum of the components
        mask = tauval > 1.0e-4
        IC_model[mask] = IC_model[mask] / (tauval[mask]) * (1. - np.exp(-tauval[mask]))
        model = SYN_model + IC_model
        return model, (en_range_list, el_dist_list)

    def _naimamodel_ind1fixed(self, pars, data):
        """
        Example set-up of the free parameters for the SSC implementation
        Index1 of the BPL is fixed as Index2 - 1 (cooling break)
        Index2 of the BPL is free
        The minimum energy and the normalization of the electron distribution are derived
        from the parameter eta_e

        Parameters
        ----------
           pars : list
             parameters of the model as list
           data : astropy.table.table.Table
             observational dataset (as astropy table) or
             if interested only in theoretical lines, astropy table
             containing only a column of energy values.
        Returns
        -------
           model : array_like
             values of the model in correspondence of the data
           electron_distribution : tuple
             electron distribution as tuple energy, electron_distribution(energy) in units of erg
        """

        eta_e = 10. ** pars[0]  # parameter 0: fraction of available energy ending in non-thermal electrons
        ebreak = 10. ** pars[1] * u.TeV  # parameter 1: linked to break energy of the electron distribution (as log10)
        alpha1 = pars[2] - 1.  # fixed to be a cooling break
        alpha2 = pars[2]  # parameter 2: high energy index of the ExponentialCutoffBrokenPowerLaw
        e_cutoff = (10. ** pars[3]) * u.TeV  # parameter 3: High energy cutoff of the electron distribution (as log10)
        bfield = 10. ** (pars[4]) * u.G  # parameter 4: Magnetic field (as log10)
        redf = 1. + self.redshift  # redshift factor
        doppler = self.gamma  # assumption of doppler boosting ~ Gamma
        size_reg = self.sizer * u.cm  # size of the region as astropy quantity
        # Volume shell where the emission takes place. The factor 9 comes from considering the shock in the ISM
        # Eq. 7 from GRB190829A paper from H.E.S.S. Collaboration
        vol = 4. * np.pi * self.sizer ** 2. * (self.sizer/(self.depthpar * self.gamma))
        # available energy in the shock. Eq. 8 from GRB190829A paper from H.E.S.S. Collaboration
        shock_energy = 2. * self.gamma ** 2. * self.density * mpc2_erg * u.erg
        eemax = e_cutoff.value * 1e13  # maximum energy of the electron distribution, based on 10*cut-off value in eV
        self.shock_energy = shock_energy
        self.eta_e = eta_e
        # ratio between magnetic field energy and shock energy
        self.eta_b = (bfield.value**2/(np.pi*8.)) / shock_energy.value
        ampl = 1. / u.eV  # temporary amplitude
        ECBPL = ExponentialCutoffBrokenPowerLaw(ampl, 1. * u.TeV, ebreak, alpha1, alpha2,
                                                e_cutoff)  # initialization of the electron distribution
        rat = eta_e * self.gamma * mpc2
        ener = np.logspace(9, np.log10(eemax), 100) * u.eV
        eldis = ECBPL(ener)
        ra = naima.utils.trapz_loglog(ener * eldis, ener) / naima.utils.trapz_loglog(eldis, ener)
        emin = rat / ra * 1e9 * u.eV  # calculation of the minimum injection energy. See detailed model explanation
        self.Emin = emin
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        # TODO: it might need an exception handling in the following line
        amplitude = ((eta_e * shock_energy * vol) / SYN.compute_We(Eemin=emin, Eemax=eemax * u.eV)) / u.eV
        ECBPL = ExponentialCutoffBrokenPowerLaw(amplitude, 1. * u.TeV, ebreak, alpha1, alpha2, e_cutoff)
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        self.Wesyn = SYN.compute_We(Eemin=emin,
                                    Eemax=eemax * u.eV)  # Computation of the total energy in the electron distribution
        # energy array to compute the target photon number density to compute IC radiation and gamma-gamma absorption
        # characteristic energy at the electron cutoff
        cutoff_charene = np.log10((synch_charene(bfield, e_cutoff)).value)
        min_synch_ene = -4  # minimum energy to start sampling the synchrotron spectrum
        bins_per_decade = 20  # 20 bins per decade to sample the synchrotron spectrum
        bins = int((cutoff_charene - min_synch_ene) * bins_per_decade)
        Esy = np.logspace(min_synch_ene, cutoff_charene + 1., bins) * u.eV
        Lsy = SYN.flux(Esy, distance=0 * u.cm)  # number of synchrotron photons per energy per time (units of 1/eV/s)
        # number density of synchrotron photons (dn/dE) units of 1/eV/cm3
        phn_sy = self.calc_photon_density(Lsy, size_reg)
        self.esycool = (synch_charene(bfield, ebreak))
        self.synchedens = naima.utils.trapz_loglog(Esy * phn_sy, Esy, axis=0).to('erg / cm3')
        # initialization of the IC component
        IC = InverseCompton(ECBPL, seed_photon_fields=[['SSC', Esy, phn_sy]], Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        # Compute the Synchrotron component
        self.synch_comp = (doppler ** 2.) * SYN.sed(data['energy'] / doppler * redf, distance=self.Dl)
        # Compute the IC component
        self.ic_comp = (doppler ** 2.) * IC.sed(data['energy'] / doppler * redf, distance=self.Dl)
        # model = (self.synch_comp+self.ic_comp) # Total model without absorption

        # Compute the optical depth in a shell of width R/(9*Gamma) after transformation of
        # the gamma ray energy of the data in the grb frame
        tauval = tau_val(data['energy'] / doppler * redf, Esy, phn_sy, self.sizer / (self.depthpar * self.gamma) * u.cm)
        # Absorption calculation with thickness of shell is R/(9Gamma) for ISM scenario. METHOD 1
        self.synch_compGG = self.synch_comp * np.exp(-tauval)
        self.ic_compGG = self.ic_comp * np.exp(-tauval)
        # model = (self.synch_compGG + self.ic_compGG) # Total model after absorption with METHOD 1

        # Absorption calculation that takes into account the fact that the gamma rays are produced
        # in the same region where they are absorbed. See Rybicki & Lightman eq. 1.29 - 1.30
        # with thickness of the shell R/(9Gamma) for ISM scenario. METHOD 2
        mask = tauval > 1.0e-4
        self.synch_compGG2 = self.synch_comp.copy()
        self.ic_compGG2 = self.ic_comp.copy()
        self.synch_compGG2[mask] = self.synch_comp[mask] / (tauval[mask]) * (1. - np.exp(-tauval[mask]))
        self.ic_compGG2[mask] = self.ic_comp[mask] / (tauval[mask]) * (1. - np.exp(-tauval[mask]))
        model = (self.synch_compGG2 + self.ic_compGG2)  # Total model after absorption with METHOD 2

        ener = np.logspace(np.log10(emin.to('GeV').value), 8,
                           500) * u.GeV  # Energy range to save the electron distribution from emin to 10^8 GeV
        eldis = ECBPL(ener)  # Compute the electron distribution
        electron_distribution = (ener, eldis)
        return model, electron_distribution  # returns model and electron distribution

    def _naimamodel_free_doppler(self, pars, data):
        """
        Similar to _naimamodel_ind1fixed, but with free doppler factor. See there for more references on the expressions

        Parameters
        ----------
           pars : list
             parameters of the model as list
           data : astropy.table.table.Table
             observational dataset (as astropy table) or
             if interested only in theoretical lines, astropy table
             containing only a column of energy values.
        Returns
        -------
           model : array_like
             values of the model in correspondence of the data
           electron_distribution : tuple
             electron distribution as tuple energy, electron_distribution(energy) in units of erg
        """

        eta_e = 10. ** pars[0]  # parameter 0: fraction of available energy ending in non-thermal electrons
        ebreak = 10. ** pars[1] * u.TeV  # parameter 1: linked to break energy of the electron distribution (as log10)
        alpha1 = pars[2] - 1.  # fixed to be a cooling break
        alpha2 = pars[2]  # parameter 2: high energy index of the ExponentialCutoffBrokenPowerLaw
        e_cutoff = (10. ** pars[3]) * u.TeV  # parameter 3: High energy cutoff of the electron distribution (as log10)
        bfield = 10. ** (pars[4]) * u.G  # parameter 4: Magnetic field (as log10)
        doppler = 10. ** (pars[5])  # parameter 5: Doppler factor
        redf = 1. + self.redshift  # redshift factor
        size_reg = self.sizer * u.cm  # size of the region as astropy quantity
        # volume shell where the emission takes place
        vol = 4. * np.pi * self.sizer ** 2. * (self.sizer / (self.depthpar * self.gamma))
        shock_energy = 2. * self.gamma ** 2. * self.density * mpc2_erg * u.erg  # available energy in the shock
        eemax = e_cutoff.value * 1e13  # maximum energy of the electron distribution, based on 10*cut-off value in eV
        self.shock_energy = shock_energy
        self.eta_e = eta_e
        self.eta_b = (bfield.value ** 2 / (np.pi * 8.)) / shock_energy.value
        ampl = 1. / u.eV  # temporary amplitude
        ECBPL = ExponentialCutoffBrokenPowerLaw(ampl, 1. * u.TeV, ebreak, alpha1, alpha2,
                                                e_cutoff)  # initialization of the electron distribution
        rat = eta_e * self.gamma * mpc2
        ener = np.logspace(9, np.log10(eemax), 100) * u.eV
        eldis = ECBPL(ener)
        ra = naima.utils.trapz_loglog(ener * eldis, ener) / naima.utils.trapz_loglog(eldis, ener)
        emin = rat / ra * 1e9 * u.eV  # calculation of the minimum injection energy. See detailed model explanation
        self.Emin = emin
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        # TODO: it might need an exception handling in the following line
        amplitude = ((eta_e * shock_energy * vol) / SYN.compute_We(Eemin=emin, Eemax=eemax * u.eV)) / u.eV
        ECBPL = ExponentialCutoffBrokenPowerLaw(amplitude, 1. * u.TeV, ebreak, alpha1, alpha2, e_cutoff)
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        self.Wesyn = SYN.compute_We(Eemin=emin,
                                    Eemax=eemax * u.eV)  # Computation of the total energy in the electron distribution
        # energy array to compute the target photon number density to compute IC radiation and gamma-gamma absorption
        # characteristic energy at the electron cutoff
        cutoff_charene = np.log10((synch_charene(bfield, e_cutoff)).value)
        min_synch_ene = -4  # minimum energy to start sampling the synchrotron spectrum
        bins_per_decade = 20  # 20 bins per decade to sample the synchrotron spectrum
        bins = int((cutoff_charene - min_synch_ene) * bins_per_decade)
        Esy = np.logspace(min_synch_ene, cutoff_charene + 1, bins) * u.eV
        Lsy = SYN.flux(Esy, distance=0 * u.cm)  # number of synchrotron photons per energy per time (units of 1/eV/s)
        # number density of synchrotron photons (dn/dE) units of 1/eV/cm3
        phn_sy = self.calc_photon_density(Lsy, size_reg)
        self.esycool = (synch_charene(bfield, ebreak))
        self.synchedens = naima.utils.trapz_loglog(Esy * phn_sy, Esy, axis=0).to('erg / cm3')
        # initialization of the IC component
        IC = InverseCompton(ECBPL, seed_photon_fields=[['SSC', Esy, phn_sy]], Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        # Compute the Synchrotron component
        self.synch_comp = (doppler ** 2.) * SYN.sed(data['energy'] / doppler * redf, distance=self.Dl)
        # Compute the IC component
        self.ic_comp = (doppler ** 2.) * IC.sed(data['energy'] / doppler * redf, distance=self.Dl)
        # model = (self.synch_comp+self.ic_comp) # Total model without absorption

        # Compute the optical depth in a shell of width R/(9*Gamma) after transformation of
        # the gamma ray energy of the data in the grb frame
        tauval = tau_val(data['energy'] / doppler * redf, Esy, phn_sy, self.sizer / (self.depthpar * self.gamma) * u.cm)
        # Absorption calculation with thickness of shell is R/(9Gamma) for ISM scenario. METHOD 1
        self.synch_compGG = self.synch_comp * np.exp(-tauval)
        self.ic_compGG = self.ic_comp * np.exp(-tauval)
        # model = (self.synch_compGG + self.ic_compGG) # Total model after absorption with METHOD 1

        # Absorption calculation that takes into account the fact that the gamma rays are produced
        # in the same region where they are absorbed
        # with thickness of the shell R/(9Gamma) for ISM scenario. METHOD 2
        mask = tauval > 1.0e-4
        self.synch_compGG2 = self.synch_comp.copy()
        self.ic_compGG2 = self.ic_comp.copy()
        self.synch_compGG2[mask] = self.synch_comp[mask] / (tauval[mask]) * (1. - np.exp(-tauval[mask]))
        self.ic_compGG2[mask] = self.ic_comp[mask] / (tauval[mask]) * (1. - np.exp(-tauval[mask]))
        model = (self.synch_compGG2 + self.ic_compGG2)  # Total model after absorption with METHOD 2

        ener = np.logspace(np.log10(emin.to('GeV').value), 8,
                           500) * u.GeV  # Energy range to save the electron distribution from emin to 10^8 GeV
        eldis = ECBPL(ener)  # Compute the electron distribution
        return model, (ener, eldis)  # returns model and electron distribution

    def _naimamodel_all_free(self, pars, data):
        """
        Similar to _naimamodel_ind1fixed,
        but with all parameters of the broken power law left free.
        See there for more details on the expressions

        Parameters
        ----------
           pars : list
             parameters of the model as list
           data : astropy.table.table.Table
             observational dataset (as astropy table) or
             if interested only in theoretical lines, astropy table
             containing only a column of energy values.
        Returns
        -------
           model : array_like
             values of the model in correspondence of the data
           electron_distribution : tuple
             electron distribution as tuple energy, electron_distribution(energy) in units of erg
        """

        eta_e = 10. ** pars[0]  # parameter 0: fraction of available energy ending in non-thermal electrons
        ebreak = 10. ** pars[1] * u.TeV  # parameter 1: break energy
        emin = (10. ** pars[2]) * u.eV  # parameter 2: minimum injection energy
        alpha1 = pars[3]               # parameter 3: low energy index
        alpha2 = pars[4]               # parameter 4: high energy index
        e_cutoff = (10. ** pars[5]) * u.TeV  # parameter 5: cutoff energy
        bfield = 10. ** (pars[6]) * u.G      # parameter 6: magnetic field
        redf = 1. + self.redshift  # redshift factor
        size_reg = self.sizer * u.cm  # size of the region as astropy quantity
        # volume shell where the emission takes place
        vol = 4. * np.pi * self.sizer ** 2. * (self.sizer / (self.depthpar * self.gamma))
        shock_energy = 2. * self.gamma ** 2. * self.density * mpc2_erg * u.erg  # available energy in the shock
        self.Emin = emin
        eemax = e_cutoff.value * 1e13  # maximum energy of the electron distribution, based on 10*cut-off value in eV
        self.shock_energy = shock_energy
        self.eta_e = eta_e
        self.eta_b = (bfield.value ** 2. / (np.pi * 8.)) / shock_energy.value
        doppler = self.gamma  # assumption that Doppler boosting ~ Gamma
        ampl = 1. / u.eV  # temporary amplitude
        ECBPL = ExponentialCutoffBrokenPowerLaw(ampl, 1. * u.TeV, ebreak, alpha1, alpha2,
                                                e_cutoff)  # initialization of the electron distribution
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        # TODO: it might need an exception handling in the following line
        amplitude = ((eta_e * shock_energy * vol) / SYN.compute_We(Eemin=emin, Eemax=eemax * u.eV)) / u.eV
        ECBPL = ExponentialCutoffBrokenPowerLaw(amplitude, 1. * u.TeV, ebreak, alpha1, alpha2, e_cutoff)
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        self.Wesyn = SYN.compute_We(Eemin=emin,
                                    Eemax=eemax * u.eV)  # Computation of the total energy in the electron distribution
        # energy array to compute the target photon number density to compute IC radiation and gamma-gamma absorption
        # characteristic energy at the electron cutoff
        cutoff_charene = np.log10((synch_charene(bfield, e_cutoff)).value)
        min_synch_ene = -4  # minimum energy to start sampling the synchrotron spectrum
        bins_per_decade = 20  # 20 bins per decade to sample the synchrotron spectrum
        bins = int((cutoff_charene - min_synch_ene) * bins_per_decade)
        Esy = np.logspace(min_synch_ene, cutoff_charene + 1, bins) * u.eV
        Lsy = SYN.flux(Esy, distance=0 * u.cm)  # number of synchrotron photons per energy per time (units of 1/eV/s)
        # number density of synchrotron photons (dn/dE) units of 1/eV/cm3
        phn_sy = self.calc_photon_density(Lsy, size_reg)
        self.esycool = (synch_charene(bfield, ebreak))
        self.synchedens = naima.utils.trapz_loglog(Esy * phn_sy, Esy, axis=0).to('erg / cm3')
        # initialization of the IC component
        IC = InverseCompton(ECBPL, seed_photon_fields=[['SSC', Esy, phn_sy]], Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        # Compute the Synchrotron component
        self.synch_comp = (doppler ** 2.) * SYN.sed(data['energy'] / doppler * redf, distance=self.Dl)
        # Compute the IC component
        self.ic_comp = (doppler ** 2.) * IC.sed(data['energy'] / doppler * redf, distance=self.Dl)
        # model = (self.synch_comp+self.ic_comp) # Total model without absorption

        # Compute the optical depth in a shell of width R/(9*Gamma) after transformation of
        # the gamma ray energy of the data in the grb frame
        tauval = tau_val(data['energy'] / doppler * redf, Esy, phn_sy, self.sizer / (self.depthpar * self.gamma) * u.cm)
        # Absorption calculation with thickness of shell is R/(9Gamma) for ISM scenario. METHOD 1
        self.synch_compGG = self.synch_comp * np.exp(-tauval)
        self.ic_compGG = self.ic_comp * np.exp(-tauval)
        # model = (self.synch_compGG + self.ic_compGG) # Total model after absorption with METHOD 1

        # Absorption calculation that takes into account the fact that the gamma rays are produced
        # in the same region where they are absorbed
        # with thickness of the shell R/(9Gamma) for ISM scenario. METHOD 2
        mask = tauval > 1.0e-4
        self.synch_compGG2 = self.synch_comp.copy()
        self.ic_compGG2 = self.ic_comp.copy()
        self.synch_compGG2[mask] = self.synch_comp[mask] / (tauval[mask]) * (1. - np.exp(-tauval[mask]))
        self.ic_compGG2[mask] = self.ic_comp[mask] / (tauval[mask]) * (1. - np.exp(-tauval[mask]))
        model = (self.synch_compGG2 + self.ic_compGG2)  # Total model after absorption with METHOD 2

        ener = np.logspace(np.log10(emin.to('GeV').value), 8,
                           500) * u.GeV  # Energy range to save the electron distribution from emin to 10^8 GeV
        eldis = ECBPL(ener)  # Compute the electron distribution
        return model, (ener, eldis)  # returns model and electron distribution

    def naimamodel_iccomps(self, pars, data, intervals, doppler=None):
        """
        Example set-up of the free parameters for the SSC implementation
        dividing the contribution of the various Synchrotron parts. See _naimamodel_ind1fixed for details on the
        expressions.

        WARNING: This function does not work if the initialization has been done with all_free = True

        Parameters
        ----------
           pars : list
             parameters of the model as list
           data : astropy.table.table.Table
             observational dataset (as astropy table) or
             if interested only in theoretical lines, astropy table
             containing only a column of energy values.
           intervals : int
             number of intervals to divide the synchrotron component
           doppler : float
             optional, doppler boosting factor for the grb
        Returns
        -------
           icsedl : list
             list of IC components in the SED
        """

        eta_e = 10. ** pars[0]  # parameter 0: fraction of available energy ending in non-thermal electrons
        ebreak = 10. ** pars[1] * u.TeV  # parameter 1: linked to break energy of the electron distribution (as log10)
        alpha1 = pars[2] - 1.  # fixed to be a cooling break
        alpha2 = pars[2]  # parameter 2: high energy index of the ExponentialCutoffBrokenPowerLaw
        e_cutoff = (10. ** pars[3]) * u.TeV  # parameter 3: High energy cutoff of the electron distribution (as log10)
        bfield = 10. ** (pars[4]) * u.G  # parameter 4: Magnetic field (as log10)
        redf = 1. + self.redshift  # redshift factor
        if doppler:
            doppler = doppler  # possible to set the doppler factor from the function argument TODO: test
        else:
            doppler = self.gamma  # assumption of doppler boosting ~ Gamma
        size_reg = self.sizer * u.cm  # size of the region as astropy quantity
        vol = 4. * np.pi * self.sizer ** 2. * (
                    self.sizer / (self.depthpar * self.gamma))  # volume shell where the emission takes place
        shock_energy = 2. * self.gamma ** 2. * self.density * mpc2.value * u.erg  # available energy in the shock
        eemax = e_cutoff.value * 1e13
        self.eta_e = eta_e
        self.eta_b = (bfield.value ** 2. / (np.pi * 8.)) / shock_energy.value
        ampl = 1. / u.eV  # temporary amplitude
        ECBPL = ExponentialCutoffBrokenPowerLaw(ampl, 1. * u.TeV, ebreak, alpha1, alpha2,
                                                e_cutoff)  # initialization of the electron distribution
        rat = eta_e * self.gamma * mpc2
        ener = np.logspace(9, np.log10(eemax), 100) * u.eV
        eldis = ECBPL(ener)
        ra = naima.utils.trapz_loglog(ener * eldis, ener) / naima.utils.trapz_loglog(eldis, ener)
        emin = rat / ra * 1e9 * u.eV  # calculation of the minimum injection energy. See detailed model explanation
        self.Emin = emin
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        amplitude = ((eta_e * shock_energy * vol) / SYN.compute_We(Eemin=emin, Eemax=eemax * u.eV)) / u.eV
        ECBPL = ExponentialCutoffBrokenPowerLaw(amplitude, 1. * u.TeV, ebreak, alpha1, alpha2, e_cutoff)
        SYN = Synchrotron(ECBPL, B=bfield, Eemin=emin, Eemax=eemax * u.eV, nEed=20)
        ener = np.linspace(-6, 2, intervals)  # set the number of wanted intervals
        Esyl = []
        Lsyl = []
        phn_syl = []
        ICl = []
        icsedl = []
        for i in range(intervals - 1):
            Esylc = np.logspace(ener[i], ener[i + 1], 100) * u.eV
            Esyl.append(Esylc)
            print("synch energy: ", ener[i], ener[i + 1])
            Lsylc = SYN.flux(Esylc, distance=0 * u.cm)
            Lsyl.append(Lsylc)
            phn_sylc = self.calc_photon_density(Lsylc, size_reg)
            phn_syl.append(phn_sylc)
            name = "SSC%i" % i
            IClc = InverseCompton(ECBPL, seed_photon_fields=[[name, Esylc, phn_sylc]], Eemin=emin, Eemax=eemax * u.eV,
                                  nEed=80)
            ICl.append(IClc)
            icsedlc = doppler ** 2 * IClc.sed(data['energy'] / doppler * redf, distance=self.Dl)
            icsedl.append(icsedlc)
        return icsedl

    def _lnprior_ind2free_wlim(self, pars):
        """
        Basic prior function where some basic parameters of the electron distribution are left free.
        The following parameters are free to vary:

           - pars[0] = log10(eta_e) (limits: [-5,0])
           - pars[1] = log10(break energy), in Tev (limits: [-6,Eblim]) where Eblim is the limit
             coming from the X-ray points (see source code). Custom made for HESS
           - pars[2] = high energy index, on linear scale (limits: [-1,5])
           - pars[3] = log10(cut-off energy), in TeV (limits: [Eblim,cutoff_limit(Bfield)])
             where cutoff_limit(Bfield) is the cut-off dictated by the synchrotron burn-off limit
           - pars[4] = log10(magnetic field), in Gauss (limits: [-3,1])

        Parameters
        ----------
          pars : list
            list of parameters passed to the model
        Returns
        -------
          prior : float
            prior probability
        """

        # EXAMPLE of break energy limit: limit of the X-ray data.
        # WARNING: The Eblim is custom made for the HESS GRB. The break in the electron will produce a break in the
        # photons that will be below the energy range of the X-ray data
        Eblim = np.log10(
            np.sqrt(
                (((((self.eblimdata['energy'][0]) * u.eV).to('erg').value / self.gamma * (1. + self.redshift)) / 3.) *
                 (2. / 3.) * 2. * np.pi * m_e ** 3. * c ** 5.) /
                (h * el * 10. ** pars[4])
            ) * erg_to_eV
        ) - 12.
        # rest of the prior
        prior0 = uniform_prior(pars[0], -5, 0)
        prior1 = uniform_prior(pars[1], -6, Eblim)
        prior2 = uniform_prior(pars[2], -1, 5)
        prior3 = uniform_prior(pars[3], Eblim, cutoff_limit(10 ** pars[4]))
        prior4 = uniform_prior(pars[4], -3, 3)
        lnprior = prior0 + prior1 + prior2 + prior3 + prior4
        return lnprior

    def _lnprior_ind2free_wlim_dopp(self, pars):
        """
        Basic prior function where some basic parameters of the electron distribution are left free,
        together with the final doppler boosting factor to return to the observer frame.
        The following parameters are free to vary:

           - pars[0] = log10(eta_e) (limits: [-5,0])
           - pars[1] = log10(break energy), in Tev (limits: [-6,Eblim]) where Eblim is the limit
             coming from the X-ray points (see source code). Custom made for HESS
           - pars[2] = high energy index, on linear scale (limits: [-1,5])
           - pars[3] = log10(cut-off energy), in TeV (limits: [Eblim,cutoff_limit(Bfield)])
             where cutoff_limit(Bfield) is the cut-off dictated by the synchrotron burn-off limit
           - pars[4] = log10(magnetic field), in Gauss (limits: [-3,1])
           - pars[5] = log10(doppler factor) (limits: [log10(grb gamma),3])

        Parameters
        ----------
          pars : list
            list of parameters passed to the model
        Returns
        -------
          prior : float
            prior probability
        """

        # EXAMPLE of break energy limit: limit of the X-ray data.
        # WARNING: The Eblim is custom made for the HESS GRB. The break in the electron will produce a break in the
        # photons that will be below the energy range of the X-ray data
        Eblim = np.log10(
            np.sqrt(
                (((((self.eblimdata['energy'][0]) * u.eV).to('erg').value / (10 ** pars[5]) *
                   (1. + self.redshift)) / 3.) *
                 (2. / 3.) * 2. * np.pi * m_e ** 3. * c ** 5.) /
                (h * el * 10. ** pars[4])
            ) * erg_to_eV
        ) - 12.
        # rest of the prior
        prior0 = uniform_prior(pars[0], -5, 0)
        prior1 = uniform_prior(pars[1], -6, Eblim)
        prior2 = uniform_prior(pars[2], -1, 5)
        prior3 = uniform_prior(pars[3], Eblim, cutoff_limit(10 ** pars[4]))
        prior4 = uniform_prior(pars[4], -3, 3)
        prior5 = uniform_prior(pars[5], np.log10(self.gamma), 5)
        lnprior = prior0 + prior1 + prior2 + prior3 + prior4 + prior5
        return lnprior

    def _lnprior_ind2free_wlim_allfree_wcooling_constrain(self, pars):
        """
        Basic prior function where most of the parameters of the electron distribution are left free
        The following parameters are free to vary:

           - pars[0] = log10(eta_e) (limits: [-5,0])
           - pars[1] = log10(break energy), in Tev (limits: [-6,Eblim]) where Eblim is the limit
             coming from the X-ray points (see source code). Custom made for HESS
           - pars[2] = log10(minimum injection energy), in eV (limits: [6,10])
           - pars[3] = low energy index, on linear scale (limits: [-1,5])
           - pars[4] = high energy index, on linear scale (limits: [-1,5])
           - pars[5] = log10(cut-off energy), in TeV (limits: [Eblim,cutoff_limit(Bfield)])
             where cutoff_limit(Bfield) is the cut-off dictated by the synchrotron burn-off limit
           - pars[6] = log10(magnetic field), in Gauss (limits: [-3,1])

        In this function there is an additional prior given by having the synchrotron cooling time of
        an electron at the break to be equal to the comoving age of the system. This prior is implemented
        through a normal prior distribution.

        Parameters
        ----------
          pars : list
            list of parameters passed to the model
        Returns
        -------
          prior : float
            prior probability
        """

        # EXAMPLE of break energy limit: limit of the X-ray data.
        # WARNING: The Eblim is custom made for the HESS GRB. The break in the electron will produce a break in the
        # photons that will be below the energy range of the X-ray data
        Eblim = np.log10(
            np.sqrt(
                (((((self.eblimdata['energy'][0]) * u.eV).to('erg').value / self.gamma * (1. + self.redshift)) / 3.) *
                 (2. / 3.) * 2. * np.pi * m_e ** 3. * c ** 5.) /
                (h * el * 10. ** pars[4])
            ) * erg_to_eV
        ) - 12.
        tcool = self.avtime * self.gamma  # age of system in comoving frame
        datacool = synch_cooltime_partene(10. ** pars[6] * u.G,
                                          10. ** pars[1] * u.TeV).value  # cooling time at break energy
        additional = normal_prior(datacool, tcool,
                                  tcool * 0.5)
        # rest of the prior
        prior0 = uniform_prior(pars[0], -5, 0)
        prior1 = uniform_prior(pars[1], -6, Eblim)
        prior2 = uniform_prior(pars[2], 6, 10)  # emin between 1MeV and 10 GeV
        prior3 = uniform_prior(pars[3], -1, 5)
        prior4 = uniform_prior(pars[4], -1, 5)
        prior5 = uniform_prior(pars[5], Eblim, cutoff_limit(10 ** pars[6]))
        prior6 = uniform_prior(pars[6], -3, 3)
        lnprior = prior0 + prior1 + prior2 + prior3 + prior4 + prior5 + prior6 + additional
        return lnprior

    def _lnprior_ind2free_wlim_allfree(self, pars):
        """
        Basic prior function where most of the parameters of the electron distribution are left free
        The following parameters are free to vary:

           - pars[0] = log10(eta_e) (limits: [-5,0])
           - pars[1] = log10(break energy), in Tev (limits: [-6,Eblim]) where Eblim is the limit
             coming from the X-ray points (see source code). Custom made for HESS
           - pars[2] = log10(minimum injection energy), in eV (limits: [6,10])
           - pars[3] = low energy index, on linear scale (limits: [-1,5])
           - pars[4] = high energy index, on linear scale (limits: [-1,5])
           - pars[5] = log10(cut-off energy), in TeV (limits: [Eblim,cutoff_limit(Bfield)])
             where cutoff_limit(Bfield) is the cut-off dictated by the synchrotron burn-off limit
           - pars[6] = log10(magnetic field), in Gauss (limits: [-3,1])

        Parameters
        ----------
          pars : list
            list of parameters passed to the model
        Returns
        -------
          prior : float
            prior probability
        """

        # EXAMPLE of break energy limit: limit of the X-ray data.
        # WARNING: The Eblim is custom made for the HESS GRB. The break in the electron will produce a break in the
        # photons that will be below the energy range of the X-ray data
        Eblim = np.log10(
            np.sqrt(
                (((((self.eblimdata['energy'][0]) * u.eV).to('erg').value / self.gamma * (1. + self.redshift)) / 3.) *
                 (2. / 3.) * 2. * np.pi * m_e ** 3. * c ** 5.) /
                (h * el * 10. ** pars[4])
            ) * erg_to_eV
        ) - 12.
        # rest of the prior
        prior0 = uniform_prior(pars[0], -5, 0)
        prior1 = uniform_prior(pars[1], -6, Eblim)
        prior2 = uniform_prior(pars[2], 6, 10)  # emin between 1MeV and 10 GeV
        prior3 = uniform_prior(pars[3], -1, 5)
        prior4 = uniform_prior(pars[4], -1, 5)
        prior5 = uniform_prior(pars[5], Eblim, cutoff_limit(10 ** pars[6]))
        prior6 = uniform_prior(pars[6], -3, 3)
        lnprior = prior0 + prior1 + prior2 + prior3 + prior4 + prior5 + prior6
        return lnprior

    def _lnprior_ind2free_wlim_wcooling_constrain(self, pars):
        """
        Basic prior function where some basic parameters of the electron distribution are left free.
        The following parameters are free to vary:

           - pars[0] = log10(eta_e) (limits: [-5,0])
           - pars[1] = log10(break energy), in Tev (limits: [-6,Eblim]) where Eblim is the limit
             coming from the X-ray points (see source code). Custom made for HESS
           - pars[2] = high energy index, on linear scale (limits: [-1,5])
           - pars[3] = log10(cut-off energy), in TeV (limits: [Eblim,cutoff_limit(Bfield)])
             where cutoff_limit(Bfield) is the cut-off dictated by the synchrotron burn-off limit
           - pars[4] = log10(magnetic field), in Gauss (limits: [-3,1])

        In this function there is an additional prior given by having the synchrotron cooling time of
        an electron at the break to be equal to the comoving age of the system. This prior is implemented
        through a normal prior distribution.

        Parameters
        ----------
          pars : list
            list of parameters passed to the model
        Returns
        -------
          prior : float
            prior probability
        """

        tcool = self.avtime * self.gamma  # age of system in comoving frame
        # cooling time at break energy
        datacool = synch_cooltime_partene(10 ** pars[4] * u.G, 10 ** pars[1] * u.TeV).value
        # EXAMPLE of break energy limit: limit of the X-ray data.
        # WARNING: The Eblim is custom made for the HESS GRB. The break in the electron will produce a break in the
        # photons that will be below the energy range of the X-ray data
        Eblim = np.log10(
            np.sqrt(
                (((((self.eblimdata['energy'][0]) * u.eV).to('erg').value / self.gamma * (1. + self.redshift)) / 3.) *
                 (2. / 3.) * 2. * np.pi * m_e ** 3. * c ** 5.) /
                (h * el * 10. ** pars[4])
            ) * erg_to_eV
        ) - 12.
        # rest of the prior
        # lower limit of the break energy (pars[1]) is the minimum injection energy.
        prior0 = uniform_prior(pars[0], -5, 0)
        prior1 = uniform_prior(pars[1], -6, Eblim)
        prior2 = uniform_prior(pars[2], -1, 5)
        prior3 = uniform_prior(pars[3], Eblim, cutoff_limit(10 ** pars[4]))
        prior4 = uniform_prior(pars[4], -3, 3)
        additional = normal_prior(datacool, tcool,
                                  tcool * 0.5)  # gaussian prior on the cooling time at break ~ age of system
        lnprior = prior0 + prior1 + prior2 + prior3 + prior4 + additional
        return lnprior

    def _lnprior_ind2free_nolim(self, pars):
        """
        Basic prior function where some basic parameters of the electron distribution are left free.
        In this function the cut-off is not limited by the synchrotron burn-off limit
        The following parameters are free to vary:

           - pars[0] = log10(eta_e) (limits: [-5,0])
           - pars[1] = log10(break energy), in Tev (limits: [-6,Eblim]) where Eblim is the limit
             coming from the X-ray points (see source code). Custom made for HESS
           - pars[2] = high energy index, on linear scale (limits: [-1,5])
           - pars[3] = log10(cut-off energy), in TeV (limits: [-3,7])
           - pars[4] = log10(magnetic field), in Gauss (limits: [-3,1])

        Parameters
        ----------
          pars : list
            list of parameters passed to the model
        Returns
        -------
          prior : float
            prior probability
        """

        # EXAMPLE of break energy limit: limit of the X-ray data.
        # WARNING: The Eblim is custom made for the HESS GRB. The break in the electron will produce a break in the
        # photons that will be below the energy range of the X-ray data
        Eblim = np.log10(
            np.sqrt(
                (((((self.eblimdata['energy'][0]) * u.eV).to('erg').value / self.gamma * (1. + self.redshift)) / 3.) *
                 (2. / 3.) * 2. * np.pi * m_e ** 3. * c ** 5.) /
                (h * el * 10. ** pars[4])
            ) * erg_to_eV
        ) - 12.
        prior0 = uniform_prior(pars[0], -5, 0)
        prior1 = uniform_prior(pars[1], -6, Eblim)
        prior2 = uniform_prior(pars[2], -1, 5)
        prior3 = uniform_prior(pars[3], -3, 7)
        prior4 = uniform_prior(pars[4], -3, 3)
        lnprior = prior0 + prior1 + prior2 + prior3 + prior4
        return lnprior

    def _lnprior_electron_test(self, pars):
        """
        Prior function for an electron spectrum made of 6 PowerLaw components.
        See model function for the details.

        Parameters
        ----------
          pars : list
            list of parameters passed to the model
        Returns
        -------
          prior : float
            prior probability
        """

        prior0 = uniform_prior(pars[0], 30, 50)
        prior1 = uniform_prior(pars[1], 30, 50)
        prior2 = uniform_prior(pars[2], 30, 50)
        prior3 = uniform_prior(pars[3], 16, 50)
        prior4 = uniform_prior(pars[4], 30, 40)
        prior5 = uniform_prior(pars[5], 10, 50)
        lnprior = prior0 + prior1 + prior2 + prior3 + prior4 + prior5
        return lnprior

    def get_Benergydensity(self):
        """
        Returns the magnetic field energy density in cgs system
        """

        bedens = (10. ** self.pars[4]) ** 2. / (8. * np.pi)  # free parameter 4 is the B field
        return bedens * u.erg / u.cm / u.cm / u.cm

    def get_Eltotenergy(self):
        """
        Returns total energy in the electron distribution
        """

        return self.Wesyn  # which is the total electron energy injected

    def run_naima(self, filename, nwalkers, nburn, nrun, threads, prefit=True):
        """
        run the naima fitting routine. Wrapper around naima.run_sampler, naima.save_run,
        and naima.save_results_table.
        Filename is the basename of the file to be saved.
        Default arguments are to run a prefit to the dataset using a ML approach.
        Beware that the ML fit might converge in a region not allowed by the parameter space

        Parameters
        ----------
          filename : string
            string with the base name of the file to be saved
          nwalkers : int
            number of parallel walkers used for the MCMC fitting
          nburn : int
            number of burn-in steps
          nrun : int
            number of steps after burn-in
          threads : int
            number of parallel threads
          prefit : bool
            If `True` performs a Maximum Likelihood fit to get a better starting point for
            for the MCMC chain (Default = True)
        Returns
        -------
          sampler : array_like
            full chain of the MCMC
        """

        sampler, pos = naima.run_sampler(data_table=self.dataset,
                                         p0=self.pars,
                                         labels=self.labels,
                                         model=self.naimamodel,
                                         prior=self.lnprior,
                                         prefit=prefit, guess=False,
                                         nwalkers=nwalkers, nburn=nburn, nrun=nrun, threads=threads)
        naima.save_run(filename=filename, sampler=sampler, clobber=True)
        naima.save_results_table(outname=filename, sampler=sampler)
        return sampler

    def integral_flux(self, emin, emax, energyflux):
        """
        Compute the integral flux (or energy flux) of of the model between emin and emax.

        Parameters
        ----------
          emin : float
            minimum energy of the interval (in eV)
          emax : float
            maximum energy of the interval (in eV)
          energyflux : bool
            boolean set to True to compute energy flux (erg/cm2/s) False for normal flux (1/cm2/s)
        Returns
        -------
          intflux : astropy.quantity
            integral flux (in units of 1/cm2/s or erg/cm2/s)
        """

        enarray = (np.logspace(np.log10(emin), np.log10(emax), 10) * u.eV).to('erg')  # in erg
        newene = Table([enarray], names=['energy'])
        model = self.naimamodel(self.pars, newene)[0]
        if energyflux:
            ednde = model / enarray
            intflux = naima.utils.trapz_loglog(ednde, enarray)
        else:
            dnde = model / enarray / enarray
            intflux = naima.utils.trapz_loglog(dnde, enarray)
        return intflux

    def quick_plot_sed(self, emin, emax, ymin, ymax):
        """
        Function for a quick plot of the model on a user specific energy range.
        If a dataset is present, this is plotted using NAIMA internal routine.

        Parameters
        ----------
          emin : float
            minimum energy of the interval (in eV)
          emax : float
            maximum energy of the interval (in eV)
          ymin : float
            minimum value for the y-axis (in erg/cm2/s)
          ymax : float
            maximum value for the y-axis (in erg/cm2/s)
        """

        bins = int(np.log10(emax/emin) * 20.)  # use 20 bins per decade
        newene = Table([np.logspace(np.log10(emin), np.log10(emax), bins) * u.eV], names=['energy'])  # energy in eV
        model = self.naimamodel(self.pars, newene)  # make sure we are computing the model for the new energy range
        f = plt.figure()
        if self.dataset:
            naima.plot_data(self.dataset, figure=f)
        plt.loglog(newene, model[0], 'k-', label='TOTAL', lw=3, alpha=0.8)
        plt.loglog(newene, self.synch_comp, 'k--', label='Synch. w/o abs.')
        plt.loglog(newene, self.ic_comp, 'k:', label='IC w/o abs.')
        plt.legend()
        plt.xlim(emin, emax)
        plt.ylim(ymin, ymax)

    def loaddataset_gammapy(self):
        """
        This function has the purpose to properly load the HESS butterfly
        to properly plot the results.
        The purpose is to have the edge of the butterfly at the beginning
        and end of the energy range in the plot.
        """
        if self.avtime < 97854:
            # full first night
            emin = 0.178  # TeV
            emax = 3.3  # TeV
            # fit PL+EBL results
            norm = 28.20e-12
            inde = -2.06
            scal = 0.5
            cova = np.array([[4.63212383e-24, 3.09859100e-14],
                             [3.09859100e-14, 9.69623830e-03]])
            covas = np.array([[(1.98 ** 2) * 1e-24, 0],
                              [0, 0.24 ** 2]])
            nbins = int(np.log10(emax / emin) * 5) + 1  ## maximum 5 bins per energy decade
        else:
            # second night
            emin = 0.178  # TeV
            emax = 1.42  # TeV
            # fit PL+EBL results
            norm = 4.79e-12
            inde = -1.86
            scal = 0.5
            cova = np.array([[1.45668716e-24, 1.46162338e-13],
                             [1.46162338e-13, 6.87092968e-02]])
            covas = np.array([[(0.24 ** 2) * 1e-24, 0],
                              [0, 0.14 ** 2]])
            nbins = 6  # int(np.log10(emax/emin)*5) ## maximum 5 bins per energy decade

        ## build the HESS butterfly starting from the covariance matrix of the fit
        enhessor2 = np.logspace(np.log10(emin), np.log10(emax), 10 * nbins)
        deltaf = list(map(lambda x: np.sqrt(np.dot(np.dot([fitfunc_N(x, scal, inde),
                                                           fitfunc_a(x, norm, inde, scal)], (cova + covas)),
                                                   [fitfunc_N(x, scal, inde),
                                                    fitfunc_a(x, norm, inde, scal)])), enhessor2))
        fakehess = Table([(enhessor2 * u.TeV).to('eV'),
                          fitfunc(enhessor2, norm, inde, scal) / (u.TeV * u.cm * u.cm * u.s),
                          (deltaf) / (u.TeV * u.cm * u.cm * u.s),
                          (deltaf) / (u.TeV * u.cm * u.cm * u.s)],
                         names=['energy', 'flux', 'flux_error_hi', 'flux_error_lo'])
        self.hessrealbf = fakehess
        enhessor2 = np.logspace(np.log10(emin), np.log10(emax), 10 * nbins)
        deltaf = list(map(lambda x: np.sqrt(np.dot(np.dot([fitfunc_N(x, scal, inde),
                                                           fitfunc_a(x, norm, inde, scal)], cova),
                                                   [fitfunc_N(x, scal, inde),
                                                    fitfunc_a(x, norm, inde, scal)])), enhessor2))
        fakehess = Table([(enhessor2 * u.TeV).to('eV'),
                          fitfunc(enhessor2, norm, inde, scal) / (u.TeV * u.cm * u.cm * u.s),
                          (deltaf) / (u.TeV * u.cm * u.cm * u.s),
                          (deltaf) / (u.TeV * u.cm * u.cm * u.s)],
                         names=['energy', 'flux', 'flux_error_hi', 'flux_error_lo'])
        self.hessrealbf_statonly = fakehess
